// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.storage.file.datalake.sas;

import com.azure.storage.common.implementation.Constants;

import java.util.Locale;

/**
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS to a path. Setting
 * a value to true means that any SAS which uses these permissions will grant permissions for that operation. It is
 * possible to construct the permissions string without this class, but the order of the permissions is particular and
 * this class guarantees correctness.
 */
public final class PathSasPermission {

    private boolean readPermission;

    private boolean addPermission;

    private boolean createPermission;

    private boolean writePermission;

    private boolean deletePermission;

    private boolean listPermission;

    private boolean movePermission;

    private boolean executePermission;

    private boolean manageOwnershipPermission;

    private boolean manageAccessControlPermission;


    /**
     * Initializes a {@code PathSasPermission} object with all fields set to false.
     */
    public PathSasPermission() {
    }

    /**
     * Creates a {@code PathSasPermission} from the specified permissions string. This method will throw an
     * {@code IllegalArgumentException} if it encounters a character that does not correspond to a valid permission.
     *
     * @param permissionString A {@code String} which represents the {@code PathSasPermission}.
     * @return A {@code PathSasPermission} generated from the given {@code String}.
     * @throws IllegalArgumentException If {@code permissionString} contains a character other than r, a, c, w, d, l, m, e,
     * o, or p.
     */
    public static PathSasPermission parse(String permissionString) {
        PathSasPermission permissions = new PathSasPermission();

        for (int i = 0; i < permissionString.length(); i++) {
            char c = permissionString.charAt(i);
            switch (c) {
                case 'r':
                    permissions.readPermission = true;
                    break;
                case 'a':
                    permissions.addPermission = true;
                    break;
                case 'c':
                    permissions.createPermission = true;
                    break;
                case 'w':
                    permissions.writePermission = true;
                    break;
                case 'd':
                    permissions.deletePermission = true;
                    break;
                case 'l':
                    permissions.listPermission = true;
                    break;
                case 'm':
                    permissions.movePermission = true;
                    break;
                case 'e':
                    permissions.executePermission = true;
                    break;
                case 'o':
                    permissions.manageOwnershipPermission = true;
                    break;
                case 'p':
                    permissions.manageAccessControlPermission = true;
                    break;
                default:
                    throw new IllegalArgumentException(
                        String.format(Locale.ROOT, Constants.ENUM_COULD_NOT_BE_PARSED_INVALID_VALUE,
                            "Permissions", permissionString, c));
            }
        }
        return permissions;
    }

    /**
     * @return the read permission status.
     */
    public boolean hasReadPermission() {
        return readPermission;
    }

    /**
     * Sets the read permission status.
     *
     * @param hasReadPermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setReadPermission(boolean hasReadPermission) {
        this.readPermission = hasReadPermission;
        return this;
    }

    /**
     * @return the add permission status.
     */
    public boolean hasAddPermission() {
        return addPermission;
    }

    /**
     * Sets the add permission status.
     *
     * @param hasAddPermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setAddPermission(boolean hasAddPermission) {
        this.addPermission = hasAddPermission;
        return this;
    }

    /**
     * @return the create permission status.
     */
    public boolean hasCreatePermission() {
        return createPermission;
    }

    /**
     * Sets the create permission status.
     *
     * @param hasCreatePermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setCreatePermission(boolean hasCreatePermission) {
        this.createPermission = hasCreatePermission;
        return this;
    }

    /**
     * @return the write permission status.
     */
    public boolean hasWritePermission() {
        return writePermission;
    }

    /**
     * Sets the write permission status.
     *
     * @param hasWritePermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setWritePermission(boolean hasWritePermission) {
        this.writePermission = hasWritePermission;
        return this;
    }

    /**
     * @return the delete permission status.
     */
    public boolean hasDeletePermission() {
        return deletePermission;
    }

    /**
     * Sets the delete permission status.
     *
     * @param hasDeletePermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setDeletePermission(boolean hasDeletePermission) {
        this.deletePermission = hasDeletePermission;
        return this;
    }

    /**
     * @return the list permission status.
     */
    public boolean hasListPermission() {
        return listPermission;
    }

    /**
     * Sets the list permission status.
     *
     * @param hasListPermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setListPermission(boolean hasListPermission) {
        this.listPermission = hasListPermission;
        return this;
    }

    /**
     * @return the move permission status.
     */
    public boolean hasMovePermission() {
        return movePermission;
    }

    /**
     * Sets the move permission status.
     *
     * @param hasMovePermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setMovePermission(boolean hasMovePermission) {
        this.movePermission = hasMovePermission;
        return this;
    }

    /**
     * @return the execute permission status.
     */
    public boolean hasExecutePermission() {
        return executePermission;
    }

    /**
     * Sets the execute permission status.
     *
     * @param hasExecutePermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setExecutePermission(boolean hasExecutePermission) {
        this.executePermission = hasExecutePermission;
        return this;
    }

    /**
     * @return the manage ownership permission status.
     */
    public boolean hasManageOwnershipPermission() {
        return manageOwnershipPermission;
    }

    /**
     * Sets the manage ownership permission status.
     *
     * @param hasManageOwnershipPermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setManageOwnershipPermission(boolean hasManageOwnershipPermission) {
        this.manageOwnershipPermission = hasManageOwnershipPermission;
        return this;
    }

    /**
     * @return the manage access control permission status.
     */
    public boolean hasManageAccessControlPermission() {
        return manageAccessControlPermission;
    }

    /**
     * Sets the manage access control permission status.
     *
     * @param hasManageAccessControlPermission Permission status to set
     * @return the updated PathSasPermission object.
     */
    public PathSasPermission setManageAccessControlPermission(boolean hasManageAccessControlPermission) {
        this.manageAccessControlPermission = hasManageAccessControlPermission;
        return this;
    }

    /**
     * Converts the given permissions to a {@code String}. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * @return A {@code String} which represents the {@code PathSasPermission}.
     */
    @Override
    public String toString() {
        // The order of the characters should be as specified here to ensure correctness:
        // https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas

        final StringBuilder builder = new StringBuilder();

        if (this.readPermission) {
            builder.append('r');
        }

        if (this.addPermission) {
            builder.append('a');
        }

        if (this.createPermission) {
            builder.append('c');
        }

        if (this.writePermission) {
            builder.append('w');
        }

        if (this.deletePermission) {
            builder.append('d');
        }

        if (this.listPermission) {
            builder.append('l');
        }

        if (this.movePermission) {
            builder.append('m');
        }

        if (this.executePermission) {
            builder.append('e');
        }

        if (this.manageOwnershipPermission) {
            builder.append('o');
        }

        if (this.manageAccessControlPermission) {
            builder.append('p');
        }

        return builder.toString();
    }
}
