// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
// Code generated by Microsoft (R) TypeSpec Code Generator.

package com.azure.security.keyvault.secrets.implementation;

import com.azure.core.annotation.BodyParam;
import com.azure.core.annotation.Delete;
import com.azure.core.annotation.ExpectedResponses;
import com.azure.core.annotation.Get;
import com.azure.core.annotation.HeaderParam;
import com.azure.core.annotation.Host;
import com.azure.core.annotation.HostParam;
import com.azure.core.annotation.Patch;
import com.azure.core.annotation.PathParam;
import com.azure.core.annotation.Post;
import com.azure.core.annotation.Put;
import com.azure.core.annotation.QueryParam;
import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceInterface;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.annotation.UnexpectedResponseExceptionType;
import com.azure.core.exception.ClientAuthenticationException;
import com.azure.core.exception.HttpResponseException;
import com.azure.core.exception.ResourceModifiedException;
import com.azure.core.exception.ResourceNotFoundException;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.HttpPipelineBuilder;
import com.azure.core.http.policy.RetryPolicy;
import com.azure.core.http.policy.UserAgentPolicy;
import com.azure.core.http.rest.PagedFlux;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.http.rest.PagedResponse;
import com.azure.core.http.rest.PagedResponseBase;
import com.azure.core.http.rest.RequestOptions;
import com.azure.core.http.rest.Response;
import com.azure.core.http.rest.RestProxy;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.serializer.JacksonAdapter;
import com.azure.core.util.serializer.SerializerAdapter;
import com.azure.security.keyvault.secrets.SecretServiceVersion;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import reactor.core.publisher.Mono;

/**
 * Initializes a new instance of the SecretClient type.
 */
public final class SecretClientImpl {
    /**
     * The proxy service used to perform REST calls.
     */
    private final SecretClientService service;

    /**
     */
    private final String vaultBaseUrl;

    /**
     * Gets.
     * 
     * @return the vaultBaseUrl value.
     */
    public String getVaultBaseUrl() {
        return this.vaultBaseUrl;
    }

    /**
     * Service version.
     */
    private final SecretServiceVersion serviceVersion;

    /**
     * Gets Service version.
     * 
     * @return the serviceVersion value.
     */
    public SecretServiceVersion getServiceVersion() {
        return this.serviceVersion;
    }

    /**
     * The HTTP pipeline to send requests through.
     */
    private final HttpPipeline httpPipeline;

    /**
     * Gets The HTTP pipeline to send requests through.
     * 
     * @return the httpPipeline value.
     */
    public HttpPipeline getHttpPipeline() {
        return this.httpPipeline;
    }

    /**
     * The serializer to serialize an object into a string.
     */
    private final SerializerAdapter serializerAdapter;

    /**
     * Gets The serializer to serialize an object into a string.
     * 
     * @return the serializerAdapter value.
     */
    public SerializerAdapter getSerializerAdapter() {
        return this.serializerAdapter;
    }

    /**
     * Initializes an instance of SecretClient client.
     * 
     * @param vaultBaseUrl
     * @param serviceVersion Service version.
     */
    public SecretClientImpl(String vaultBaseUrl, SecretServiceVersion serviceVersion) {
        this(new HttpPipelineBuilder().policies(new UserAgentPolicy(), new RetryPolicy()).build(),
            JacksonAdapter.createDefaultSerializerAdapter(), vaultBaseUrl, serviceVersion);
    }

    /**
     * Initializes an instance of SecretClient client.
     * 
     * @param httpPipeline The HTTP pipeline to send requests through.
     * @param vaultBaseUrl
     * @param serviceVersion Service version.
     */
    public SecretClientImpl(HttpPipeline httpPipeline, String vaultBaseUrl, SecretServiceVersion serviceVersion) {
        this(httpPipeline, JacksonAdapter.createDefaultSerializerAdapter(), vaultBaseUrl, serviceVersion);
    }

    /**
     * Initializes an instance of SecretClient client.
     * 
     * @param httpPipeline The HTTP pipeline to send requests through.
     * @param serializerAdapter The serializer to serialize an object into a string.
     * @param vaultBaseUrl
     * @param serviceVersion Service version.
     */
    public SecretClientImpl(HttpPipeline httpPipeline, SerializerAdapter serializerAdapter, String vaultBaseUrl,
        SecretServiceVersion serviceVersion) {
        this.httpPipeline = httpPipeline;
        this.serializerAdapter = serializerAdapter;
        this.vaultBaseUrl = vaultBaseUrl;
        this.serviceVersion = serviceVersion;
        this.service = RestProxy.create(SecretClientService.class, this.httpPipeline, this.getSerializerAdapter());
    }

    /**
     * The interface defining all the services for SecretClient to be used by the proxy service to perform REST calls.
     */
    @Host("{vaultBaseUrl}")
    @ServiceInterface(name = "SecretClient")
    public interface SecretClientService {
        @Put("/secrets/{secret-name}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> setSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Content-Type") String contentType, @HeaderParam("Accept") String accept,
            @BodyParam("application/json") BinaryData parameters, RequestOptions requestOptions, Context context);

        @Put("/secrets/{secret-name}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> setSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Content-Type") String contentType, @HeaderParam("Accept") String accept,
            @BodyParam("application/json") BinaryData parameters, RequestOptions requestOptions, Context context);

        @Delete("/secrets/{secret-name}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> deleteSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Delete("/secrets/{secret-name}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> deleteSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Patch("/secrets/{secret-name}/{secret-version}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> updateSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @PathParam("secret-version") String secretVersion, @HeaderParam("Content-Type") String contentType,
            @HeaderParam("Accept") String accept, @BodyParam("application/json") BinaryData parameters,
            RequestOptions requestOptions, Context context);

        @Patch("/secrets/{secret-name}/{secret-version}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> updateSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @PathParam("secret-version") String secretVersion, @HeaderParam("Content-Type") String contentType,
            @HeaderParam("Accept") String accept, @BodyParam("application/json") BinaryData parameters,
            RequestOptions requestOptions, Context context);

        @Get("/secrets/{secret-name}/{secret-version}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @PathParam("secret-version") String secretVersion, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/secrets/{secret-name}/{secret-version}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @PathParam("secret-version") String secretVersion, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/secrets")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getSecrets(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/secrets")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getSecretsSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/secrets/{secret-name}/versions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getSecretVersions(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Get("/secrets/{secret-name}/versions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getSecretVersionsSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Get("/deletedsecrets")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getDeletedSecrets(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/deletedsecrets")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getDeletedSecretsSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/deletedsecrets/{secret-name}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getDeletedSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Get("/deletedsecrets/{secret-name}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getDeletedSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Delete("/deletedsecrets/{secret-name}")
        @ExpectedResponses({ 204 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<Void>> purgeDeletedSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Delete("/deletedsecrets/{secret-name}")
        @ExpectedResponses({ 204 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<Void> purgeDeletedSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Post("/deletedsecrets/{secret-name}/recover")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> recoverDeletedSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Post("/deletedsecrets/{secret-name}/recover")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> recoverDeletedSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Post("/secrets/{secret-name}/backup")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> backupSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Post("/secrets/{secret-name}/backup")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> backupSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @PathParam("secret-name") String secretName,
            @HeaderParam("Accept") String accept, RequestOptions requestOptions, Context context);

        @Post("/secrets/restore")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> restoreSecret(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @HeaderParam("Content-Type") String contentType,
            @HeaderParam("Accept") String accept, @BodyParam("application/json") BinaryData parameters,
            RequestOptions requestOptions, Context context);

        @Post("/secrets/restore")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> restoreSecretSync(@HostParam("vaultBaseUrl") String vaultBaseUrl,
            @QueryParam("api-version") String apiVersion, @HeaderParam("Content-Type") String contentType,
            @HeaderParam("Accept") String accept, @BodyParam("application/json") BinaryData parameters,
            RequestOptions requestOptions, Context context);

        @Get("{nextLink}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getSecretsNext(@PathParam(value = "nextLink", encoded = true) String nextLink,
            @HostParam("vaultBaseUrl") String vaultBaseUrl, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("{nextLink}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getSecretsNextSync(@PathParam(value = "nextLink", encoded = true) String nextLink,
            @HostParam("vaultBaseUrl") String vaultBaseUrl, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("{nextLink}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getSecretVersionsNext(@PathParam(value = "nextLink", encoded = true) String nextLink,
            @HostParam("vaultBaseUrl") String vaultBaseUrl, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("{nextLink}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getSecretVersionsNextSync(@PathParam(value = "nextLink", encoded = true) String nextLink,
            @HostParam("vaultBaseUrl") String vaultBaseUrl, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("{nextLink}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getDeletedSecretsNext(@PathParam(value = "nextLink", encoded = true) String nextLink,
            @HostParam("vaultBaseUrl") String vaultBaseUrl, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("{nextLink}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getDeletedSecretsNextSync(@PathParam(value = "nextLink", encoded = true) String nextLink,
            @HostParam("vaultBaseUrl") String vaultBaseUrl, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);
    }

    /**
     * Sets a secret in a specified key vault.
     * 
     * The SET operation adds a secret to the Azure Key Vault. If the named secret already exists, Azure Key Vault
     * creates a new version of that secret. This operation requires the secrets/set permission.
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Required)
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     * }
     * }
     * </pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret. The value you provide may be copied globally for the purpose of running
     * the service. The value provided should not include personally identifiable or sensitive information.
     * @param parameters The parameters for setting the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response} on successful
     * completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> setSecretWithResponseAsync(String secretName, BinaryData parameters,
        RequestOptions requestOptions) {
        final String contentType = "application/json";
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.setSecret(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(),
                secretName, contentType, accept, parameters, requestOptions, context));
    }

    /**
     * Sets a secret in a specified key vault.
     * 
     * The SET operation adds a secret to the Azure Key Vault. If the named secret already exists, Azure Key Vault
     * creates a new version of that secret. This operation requires the secrets/set permission.
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Required)
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     * }
     * }
     * </pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret. The value you provide may be copied globally for the purpose of running
     * the service. The value provided should not include personally identifiable or sensitive information.
     * @param parameters The parameters for setting the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> setSecretWithResponse(String secretName, BinaryData parameters,
        RequestOptions requestOptions) {
        final String contentType = "application/json";
        final String accept = "application/json";
        return service.setSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            contentType, accept, parameters, requestOptions, Context.NONE);
    }

    /**
     * Deletes a secret from a specified key vault.
     * 
     * The DELETE operation applies to any secret stored in Azure Key Vault. DELETE cannot be applied to an individual
     * version of a secret. This operation requires the secrets/delete permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a Deleted Secret consisting of its previous id, attributes and its tags, as well as information on when
     * it will be purged along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> deleteSecretWithResponseAsync(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.deleteSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, accept, requestOptions, context));
    }

    /**
     * Deletes a secret from a specified key vault.
     * 
     * The DELETE operation applies to any secret stored in Azure Key Vault. DELETE cannot be applied to an individual
     * version of a secret. This operation requires the secrets/delete permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a Deleted Secret consisting of its previous id, attributes and its tags, as well as information on when
     * it will be purged along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> deleteSecretWithResponse(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.deleteSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            accept, requestOptions, Context.NONE);
    }

    /**
     * Updates the attributes associated with a specified secret in a given key vault.
     * 
     * The UPDATE operation changes specified attributes of an existing stored secret. Attributes that are not specified
     * in the request are left unchanged. The value of a secret itself cannot be changed. This operation requires the
     * secrets/set permission.
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     * }
     * }
     * </pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param secretVersion The version of the secret.
     * @param parameters The parameters for update secret operation.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response} on successful
     * completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> updateSecretWithResponseAsync(String secretName, String secretVersion,
        BinaryData parameters, RequestOptions requestOptions) {
        final String contentType = "application/json";
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.updateSecret(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(),
                secretName, secretVersion, contentType, accept, parameters, requestOptions, context));
    }

    /**
     * Updates the attributes associated with a specified secret in a given key vault.
     * 
     * The UPDATE operation changes specified attributes of an existing stored secret. Attributes that are not specified
     * in the request are left unchanged. The value of a secret itself cannot be changed. This operation requires the
     * secrets/set permission.
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     * }
     * }
     * </pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param secretVersion The version of the secret.
     * @param parameters The parameters for update secret operation.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> updateSecretWithResponse(String secretName, String secretVersion, BinaryData parameters,
        RequestOptions requestOptions) {
        final String contentType = "application/json";
        final String accept = "application/json";
        return service.updateSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            secretVersion, contentType, accept, parameters, requestOptions, Context.NONE);
    }

    /**
     * Get a specified secret from a given key vault.
     * 
     * The GET operation is applicable to any secret stored in Azure Key Vault. This operation requires the secrets/get
     * permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param secretVersion The version of the secret. This URI fragment is optional. If not specified, the latest
     * version of the secret is returned.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a specified secret from a given key vault.
     * 
     * The GET operation is applicable to any secret stored in Azure Key Vault along with {@link Response} on successful
     * completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getSecretWithResponseAsync(String secretName, String secretVersion,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.getSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, secretVersion, accept, requestOptions, context));
    }

    /**
     * Get a specified secret from a given key vault.
     * 
     * The GET operation is applicable to any secret stored in Azure Key Vault. This operation requires the secrets/get
     * permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param secretVersion The version of the secret. This URI fragment is optional. If not specified, the latest
     * version of the secret is returned.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a specified secret from a given key vault.
     * 
     * The GET operation is applicable to any secret stored in Azure Key Vault along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getSecretWithResponse(String secretName, String secretVersion,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            secretVersion, accept, requestOptions, Context.NONE);
    }

    /**
     * List secrets in a specified key vault.
     * 
     * The Get Secrets operation is applicable to the entire vault. However, only the base secret identifier and its
     * attributes are provided in the response. Individual secret versions are not listed in the response. This
     * operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private Mono<PagedResponse<BinaryData>> getSecretsSinglePageAsync(RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.getSecrets(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(),
                accept, requestOptions, context))
            .map(res -> new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
                getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null));
    }

    /**
     * List secrets in a specified key vault.
     * 
     * The Get Secrets operation is applicable to the entire vault. However, only the base secret identifier and its
     * attributes are provided in the response. Individual secret versions are not listed in the response. This
     * operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result as paginated response with {@link PagedFlux}.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedFlux<BinaryData> getSecretsAsync(RequestOptions requestOptions) {
        RequestOptions requestOptionsForNextPage = new RequestOptions();
        requestOptionsForNextPage.setContext(
            requestOptions != null && requestOptions.getContext() != null ? requestOptions.getContext() : Context.NONE);
        return new PagedFlux<>(() -> getSecretsSinglePageAsync(requestOptions),
            nextLink -> getSecretsNextSinglePageAsync(nextLink, requestOptionsForNextPage));
    }

    /**
     * List secrets in a specified key vault.
     * 
     * The Get Secrets operation is applicable to the entire vault. However, only the base secret identifier and its
     * attributes are provided in the response. Individual secret versions are not listed in the response. This
     * operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private PagedResponse<BinaryData> getSecretsSinglePage(RequestOptions requestOptions) {
        final String accept = "application/json";
        Response<BinaryData> res = service.getSecretsSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(),
            accept, requestOptions, Context.NONE);
        return new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
            getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null);
    }

    /**
     * List secrets in a specified key vault.
     * 
     * The Get Secrets operation is applicable to the entire vault. However, only the base secret identifier and its
     * attributes are provided in the response. Individual secret versions are not listed in the response. This
     * operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result as paginated response with {@link PagedIterable}.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<BinaryData> getSecrets(RequestOptions requestOptions) {
        RequestOptions requestOptionsForNextPage = new RequestOptions();
        requestOptionsForNextPage.setContext(
            requestOptions != null && requestOptions.getContext() != null ? requestOptions.getContext() : Context.NONE);
        return new PagedIterable<>(() -> getSecretsSinglePage(requestOptions),
            nextLink -> getSecretsNextSinglePage(nextLink, requestOptionsForNextPage));
    }

    /**
     * List all versions of the specified secret.
     * 
     * The full secret identifier and attributes are provided in the response. No values are returned for the secrets.
     * This operations requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private Mono<PagedResponse<BinaryData>> getSecretVersionsSinglePageAsync(String secretName,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.getSecretVersions(this.getVaultBaseUrl(),
                this.getServiceVersion().getVersion(), secretName, accept, requestOptions, context))
            .map(res -> new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
                getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null));
    }

    /**
     * List all versions of the specified secret.
     * 
     * The full secret identifier and attributes are provided in the response. No values are returned for the secrets.
     * This operations requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result as paginated response with {@link PagedFlux}.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedFlux<BinaryData> getSecretVersionsAsync(String secretName, RequestOptions requestOptions) {
        RequestOptions requestOptionsForNextPage = new RequestOptions();
        requestOptionsForNextPage.setContext(
            requestOptions != null && requestOptions.getContext() != null ? requestOptions.getContext() : Context.NONE);
        return new PagedFlux<>(() -> getSecretVersionsSinglePageAsync(secretName, requestOptions),
            nextLink -> getSecretVersionsNextSinglePageAsync(nextLink, requestOptionsForNextPage));
    }

    /**
     * List all versions of the specified secret.
     * 
     * The full secret identifier and attributes are provided in the response. No values are returned for the secrets.
     * This operations requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private PagedResponse<BinaryData> getSecretVersionsSinglePage(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        Response<BinaryData> res = service.getSecretVersionsSync(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, accept, requestOptions, Context.NONE);
        return new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
            getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null);
    }

    /**
     * List all versions of the specified secret.
     * 
     * The full secret identifier and attributes are provided in the response. No values are returned for the secrets.
     * This operations requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result as paginated response with {@link PagedIterable}.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<BinaryData> getSecretVersions(String secretName, RequestOptions requestOptions) {
        RequestOptions requestOptionsForNextPage = new RequestOptions();
        requestOptionsForNextPage.setContext(
            requestOptions != null && requestOptions.getContext() != null ? requestOptions.getContext() : Context.NONE);
        return new PagedIterable<>(() -> getSecretVersionsSinglePage(secretName, requestOptions),
            nextLink -> getSecretVersionsNextSinglePage(nextLink, requestOptionsForNextPage));
    }

    /**
     * Lists deleted secrets for the specified vault.
     * 
     * The Get Deleted Secrets operation returns the secrets that have been deleted for a vault enabled for soft-delete.
     * This operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the deleted secret list result along with {@link PagedResponse} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private Mono<PagedResponse<BinaryData>> getDeletedSecretsSinglePageAsync(RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.getDeletedSecrets(this.getVaultBaseUrl(),
                this.getServiceVersion().getVersion(), accept, requestOptions, context))
            .map(res -> new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
                getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null));
    }

    /**
     * Lists deleted secrets for the specified vault.
     * 
     * The Get Deleted Secrets operation returns the secrets that have been deleted for a vault enabled for soft-delete.
     * This operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the deleted secret list result as paginated response with {@link PagedFlux}.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedFlux<BinaryData> getDeletedSecretsAsync(RequestOptions requestOptions) {
        RequestOptions requestOptionsForNextPage = new RequestOptions();
        requestOptionsForNextPage.setContext(
            requestOptions != null && requestOptions.getContext() != null ? requestOptions.getContext() : Context.NONE);
        return new PagedFlux<>(() -> getDeletedSecretsSinglePageAsync(requestOptions),
            nextLink -> getDeletedSecretsNextSinglePageAsync(nextLink, requestOptionsForNextPage));
    }

    /**
     * Lists deleted secrets for the specified vault.
     * 
     * The Get Deleted Secrets operation returns the secrets that have been deleted for a vault enabled for soft-delete.
     * This operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the deleted secret list result along with {@link PagedResponse}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private PagedResponse<BinaryData> getDeletedSecretsSinglePage(RequestOptions requestOptions) {
        final String accept = "application/json";
        Response<BinaryData> res = service.getDeletedSecretsSync(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), accept, requestOptions, Context.NONE);
        return new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
            getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null);
    }

    /**
     * Lists deleted secrets for the specified vault.
     * 
     * The Get Deleted Secrets operation returns the secrets that have been deleted for a vault enabled for soft-delete.
     * This operation requires the secrets/list permission.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>maxresults</td><td>Integer</td><td>No</td><td>Maximum number of results to return in a page. If not
     * specified the service will return up to 25 results.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the deleted secret list result as paginated response with {@link PagedIterable}.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<BinaryData> getDeletedSecrets(RequestOptions requestOptions) {
        RequestOptions requestOptionsForNextPage = new RequestOptions();
        requestOptionsForNextPage.setContext(
            requestOptions != null && requestOptions.getContext() != null ? requestOptions.getContext() : Context.NONE);
        return new PagedIterable<>(() -> getDeletedSecretsSinglePage(requestOptions),
            nextLink -> getDeletedSecretsNextSinglePage(nextLink, requestOptionsForNextPage));
    }

    /**
     * Gets the specified deleted secret.
     * 
     * The Get Deleted Secret operation returns the specified deleted secret along with its attributes. This operation
     * requires the secrets/get permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the specified deleted secret.
     * 
     * The Get Deleted Secret operation returns the specified deleted secret along with its attributes along with
     * {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getDeletedSecretWithResponseAsync(String secretName,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.getDeletedSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, accept, requestOptions, context));
    }

    /**
     * Gets the specified deleted secret.
     * 
     * The Get Deleted Secret operation returns the specified deleted secret along with its attributes. This operation
     * requires the secrets/get permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the specified deleted secret.
     * 
     * The Get Deleted Secret operation returns the specified deleted secret along with its attributes along with
     * {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getDeletedSecretWithResponse(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getDeletedSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            accept, requestOptions, Context.NONE);
    }

    /**
     * Permanently deletes the specified secret.
     * 
     * The purge deleted secret operation removes the secret permanently, without the possibility of recovery. This
     * operation can only be enabled on a soft-delete enabled vault. This operation requires the secrets/purge
     * permission.
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<Void>> purgeDeletedSecretWithResponseAsync(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.purgeDeletedSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, accept, requestOptions, context));
    }

    /**
     * Permanently deletes the specified secret.
     * 
     * The purge deleted secret operation removes the secret permanently, without the possibility of recovery. This
     * operation can only be enabled on a soft-delete enabled vault. This operation requires the secrets/purge
     * permission.
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<Void> purgeDeletedSecretWithResponse(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.purgeDeletedSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            accept, requestOptions, Context.NONE);
    }

    /**
     * Recovers the deleted secret to the latest version.
     * 
     * Recovers the deleted secret in the specified vault. This operation can only be performed on a soft-delete enabled
     * vault. This operation requires the secrets/recover permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the deleted secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response} on successful
     * completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> recoverDeletedSecretWithResponseAsync(String secretName,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.recoverDeletedSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, accept, requestOptions, context));
    }

    /**
     * Recovers the deleted secret to the latest version.
     * 
     * Recovers the deleted secret in the specified vault. This operation can only be performed on a soft-delete enabled
     * vault. This operation requires the secrets/recover permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the deleted secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> recoverDeletedSecretWithResponse(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.recoverDeletedSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(),
            secretName, accept, requestOptions, Context.NONE);
    }

    /**
     * Backs up the specified secret.
     * 
     * Requests that a backup of the specified secret be downloaded to the client. All versions of the secret will be
     * downloaded. This operation requires the secrets/backup permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: Base64Url (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the backup secret result, containing the backup blob along with {@link Response} on successful completion
     * of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> backupSecretWithResponseAsync(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.backupSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), secretName, accept, requestOptions, context));
    }

    /**
     * Backs up the specified secret.
     * 
     * Requests that a backup of the specified secret be downloaded to the client. All versions of the secret will be
     * downloaded. This operation requires the secrets/backup permission.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: Base64Url (Optional)
     * }
     * }
     * </pre>
     * 
     * @param secretName The name of the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the backup secret result, containing the backup blob along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> backupSecretWithResponse(String secretName, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.backupSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), secretName,
            accept, requestOptions, Context.NONE);
    }

    /**
     * Restores a backed up secret to a vault.
     * 
     * Restores a backed up secret, and all its versions, to a vault. This operation requires the secrets/restore
     * permission.
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: Base64Url (Required)
     * }
     * }
     * </pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param parameters The parameters to restore the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response} on successful
     * completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> restoreSecretWithResponseAsync(BinaryData parameters,
        RequestOptions requestOptions) {
        final String contentType = "application/json";
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.restoreSecret(this.getVaultBaseUrl(),
            this.getServiceVersion().getVersion(), contentType, accept, parameters, requestOptions, context));
    }

    /**
     * Restores a backed up secret to a vault.
     * 
     * Restores a backed up secret, and all its versions, to a vault. This operation requires the secrets/restore
     * permission.
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: Base64Url (Required)
     * }
     * }
     * </pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     value: String (Optional)
     *     id: String (Optional)
     *     contentType: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     kid: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param parameters The parameters to restore the secret.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return a secret consisting of a value, id and its attributes along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> restoreSecretWithResponse(BinaryData parameters, RequestOptions requestOptions) {
        final String contentType = "application/json";
        final String accept = "application/json";
        return service.restoreSecretSync(this.getVaultBaseUrl(), this.getServiceVersion().getVersion(), contentType,
            accept, parameters, requestOptions, Context.NONE);
    }

    /**
     * List secrets in a specified key vault.
     * 
     * Get the next page of items.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param nextLink The URL to get the next list of items.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private Mono<PagedResponse<BinaryData>> getSecretsNextSinglePageAsync(String nextLink,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(
                context -> service.getSecretsNext(nextLink, this.getVaultBaseUrl(), accept, requestOptions, context))
            .map(res -> new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
                getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null));
    }

    /**
     * List secrets in a specified key vault.
     * 
     * Get the next page of items.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param nextLink The URL to get the next list of items.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private PagedResponse<BinaryData> getSecretsNextSinglePage(String nextLink, RequestOptions requestOptions) {
        final String accept = "application/json";
        Response<BinaryData> res
            = service.getSecretsNextSync(nextLink, this.getVaultBaseUrl(), accept, requestOptions, Context.NONE);
        return new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
            getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null);
    }

    /**
     * List all versions of the specified secret.
     * 
     * Get the next page of items.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param nextLink The URL to get the next list of items.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private Mono<PagedResponse<BinaryData>> getSecretVersionsNextSinglePageAsync(String nextLink,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(
            context -> service.getSecretVersionsNext(nextLink, this.getVaultBaseUrl(), accept, requestOptions, context))
            .map(res -> new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
                getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null));
    }

    /**
     * List all versions of the specified secret.
     * 
     * Get the next page of items.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     * }
     * }
     * </pre>
     * 
     * @param nextLink The URL to get the next list of items.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the secret list result along with {@link PagedResponse}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private PagedResponse<BinaryData> getSecretVersionsNextSinglePage(String nextLink, RequestOptions requestOptions) {
        final String accept = "application/json";
        Response<BinaryData> res
            = service.getSecretVersionsNextSync(nextLink, this.getVaultBaseUrl(), accept, requestOptions, Context.NONE);
        return new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
            getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null);
    }

    /**
     * Lists deleted secrets for the specified vault.
     * 
     * Get the next page of items.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param nextLink The URL to get the next list of items.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the deleted secret list result along with {@link PagedResponse} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private Mono<PagedResponse<BinaryData>> getDeletedSecretsNextSinglePageAsync(String nextLink,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(
            context -> service.getDeletedSecretsNext(nextLink, this.getVaultBaseUrl(), accept, requestOptions, context))
            .map(res -> new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
                getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null));
    }

    /**
     * Lists deleted secrets for the specified vault.
     * 
     * Get the next page of items.
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>
     * {@code
     * {
     *     id: String (Optional)
     *     attributes (Optional): {
     *         enabled: Boolean (Optional)
     *         nbf: Long (Optional)
     *         exp: Long (Optional)
     *         created: Long (Optional)
     *         updated: Long (Optional)
     *         recoverableDays: Integer (Optional)
     *         recoveryLevel: String(Purgeable/Recoverable+Purgeable/Recoverable/Recoverable+ProtectedSubscription/CustomizedRecoverable+Purgeable/CustomizedRecoverable/CustomizedRecoverable+ProtectedSubscription) (Optional)
     *     }
     *     tags (Optional): {
     *         String: String (Required)
     *     }
     *     contentType: String (Optional)
     *     managed: Boolean (Optional)
     *     recoveryId: String (Optional)
     *     scheduledPurgeDate: Long (Optional)
     *     deletedDate: Long (Optional)
     * }
     * }
     * </pre>
     * 
     * @param nextLink The URL to get the next list of items.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the deleted secret list result along with {@link PagedResponse}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    private PagedResponse<BinaryData> getDeletedSecretsNextSinglePage(String nextLink, RequestOptions requestOptions) {
        final String accept = "application/json";
        Response<BinaryData> res
            = service.getDeletedSecretsNextSync(nextLink, this.getVaultBaseUrl(), accept, requestOptions, Context.NONE);
        return new PagedResponseBase<>(res.getRequest(), res.getStatusCode(), res.getHeaders(),
            getValues(res.getValue(), "value"), getNextLink(res.getValue(), "nextLink"), null);
    }

    private List<BinaryData> getValues(BinaryData binaryData, String path) {
        try {
            Map<?, ?> obj = binaryData.toObject(Map.class);
            List<?> values = (List<?>) obj.get(path);
            return values.stream().map(BinaryData::fromObject).collect(Collectors.toList());
        } catch (RuntimeException e) {
            return null;
        }
    }

    private String getNextLink(BinaryData binaryData, String path) {
        try {
            Map<?, ?> obj = binaryData.toObject(Map.class);
            return (String) obj.get(path);
        } catch (RuntimeException e) {
            return null;
        }
    }
}
