/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation;

import com.azure.cosmos.BridgeInternal;
import com.azure.cosmos.CosmosDiagnostics;
import com.azure.cosmos.CosmosException;
import com.azure.cosmos.ThrottlingRetryOptions;
import com.azure.cosmos.implementation.DiagnosticsClientContext;
import com.azure.cosmos.implementation.DocumentClientRetryPolicy;
import com.azure.cosmos.implementation.Exceptions;
import com.azure.cosmos.implementation.GlobalEndpointManager;
import com.azure.cosmos.implementation.OperationType;
import com.azure.cosmos.implementation.ResourceThrottleRetryPolicy;
import com.azure.cosmos.implementation.RxDocumentServiceRequest;
import com.azure.cosmos.implementation.ServiceUnavailableException;
import com.azure.cosmos.implementation.ShouldRetryResult;
import com.azure.cosmos.implementation.Utils;
import com.azure.cosmos.implementation.apachecommons.collections.list.UnmodifiableList;
import com.azure.cosmos.implementation.apachecommons.lang.StringUtils;
import com.azure.cosmos.implementation.caches.RxCollectionCache;
import com.azure.cosmos.implementation.directconnectivity.WebExceptionUtility;
import com.azure.cosmos.implementation.faultinjection.FaultInjectionRequestContext;
import java.net.URI;
import java.time.Duration;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class ClientRetryPolicy
extends DocumentClientRetryPolicy {
    private static final Logger logger = LoggerFactory.getLogger(ClientRetryPolicy.class);
    static final int RetryIntervalInMS = 1000;
    static final int MaxRetryCount = 120;
    private static final int MaxServiceUnavailableRetryCount = 1;
    private final DocumentClientRetryPolicy throttlingRetry;
    private final GlobalEndpointManager globalEndpointManager;
    private final boolean enableEndpointDiscovery;
    private int failoverRetryCount;
    private int sessionTokenRetryCount;
    private int staleContainerRetryCount;
    private boolean isReadRequest;
    private boolean canUseMultipleWriteLocations;
    private URI locationEndpoint;
    private RetryContext retryContext;
    private CosmosDiagnostics cosmosDiagnostics;
    private AtomicInteger cnt = new AtomicInteger(0);
    private int serviceUnavailableRetryCount;
    private RxDocumentServiceRequest request;
    private RxCollectionCache rxCollectionCache;
    private final FaultInjectionRequestContext faultInjectionRequestContext;

    public ClientRetryPolicy(DiagnosticsClientContext diagnosticsClientContext, GlobalEndpointManager globalEndpointManager, boolean enableEndpointDiscovery, ThrottlingRetryOptions throttlingRetryOptions, RxCollectionCache rxCollectionCache) {
        this.globalEndpointManager = globalEndpointManager;
        this.failoverRetryCount = 0;
        this.enableEndpointDiscovery = enableEndpointDiscovery;
        this.sessionTokenRetryCount = 0;
        this.staleContainerRetryCount = 0;
        this.canUseMultipleWriteLocations = false;
        this.cosmosDiagnostics = diagnosticsClientContext.createDiagnostics();
        this.throttlingRetry = new ResourceThrottleRetryPolicy(throttlingRetryOptions.getMaxRetryAttemptsOnThrottledRequests(), throttlingRetryOptions.getMaxRetryWaitTime(), BridgeInternal.getRetryContext(this.getCosmosDiagnostics()), false);
        this.rxCollectionCache = rxCollectionCache;
        this.faultInjectionRequestContext = new FaultInjectionRequestContext();
    }

    @Override
    public Mono<ShouldRetryResult> shouldRetry(Exception e) {
        logger.debug("retry count {}, isReadRequest {}, canUseMultipleWriteLocations {}, due to failure:", new Object[]{this.cnt.incrementAndGet(), this.isReadRequest, this.canUseMultipleWriteLocations, e});
        if (this.locationEndpoint == null) {
            logger.error("locationEndpoint is null because ClientRetryPolicy::onBeforeRequest(.) is not invoked, probably request creation failed due to invalid options, serialization setting, etc.");
            return Mono.just((Object)ShouldRetryResult.error(e));
        }
        this.retryContext = null;
        CosmosException clientException = Utils.as(e, CosmosException.class);
        if (clientException != null && clientException.getDiagnostics() != null) {
            this.cosmosDiagnostics = clientException.getDiagnostics();
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 3)) {
            logger.warn("Endpoint not writable. Will refresh cache and retry ", (Throwable)e);
            return this.shouldRetryOnEndpointFailureAsync(false, true, false);
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 1008) && this.isReadRequest) {
            logger.warn("Endpoint not available for reads. Will refresh cache and retry. ", (Throwable)e);
            return this.shouldRetryOnEndpointFailureAsync(true, false, false);
        }
        if (WebExceptionUtility.isNetworkFailure(e)) {
            if (clientException != null && Exceptions.isSubStatusCode(clientException, 10001)) {
                if (this.isReadRequest || WebExceptionUtility.isWebExceptionRetriable(e)) {
                    logger.warn("Gateway endpoint not reachable. Will refresh cache and retry. ", (Throwable)e);
                    return this.shouldRetryOnEndpointFailureAsync(this.isReadRequest, false, true);
                }
                return this.shouldNotRetryOnEndpointFailureAsync(this.isReadRequest, false, false);
            }
            if (clientException != null && WebExceptionUtility.isReadTimeoutException((Exception)((Object)clientException)) && Exceptions.isSubStatusCode(clientException, 10002)) {
                return this.shouldRetryOnGatewayTimeout();
            }
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 404) && Exceptions.isSubStatusCode(clientException, 1002)) {
            return Mono.just((Object)this.shouldRetryOnSessionNotAvailable(this.request));
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 400) && Exceptions.isSubStatusCode(clientException, 1024)) {
            return this.shouldRetryOnStaleContainer();
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 503)) {
            boolean isWebExceptionRetriable = WebExceptionUtility.isWebExceptionRetriable(e);
            logger.warn("Service unavailable - IsReadRequest {}, IsWebExceptionRetriable {}, NonIdempotentWriteRetriesEnabled {}", new Object[]{this.isReadRequest, isWebExceptionRetriable, this.request.getNonIdempotentWriteRetriesEnabled(), e});
            return this.shouldRetryOnBackendServiceUnavailableAsync(this.isReadRequest, isWebExceptionRetriable, this.request.getNonIdempotentWriteRetriesEnabled(), clientException);
        }
        return this.throttlingRetry.shouldRetry(e);
    }

    private boolean canGatewayRequestFailoverOnTimeout(RxDocumentServiceRequest request) {
        if (request.getOperationType() == OperationType.QueryPlan) {
            return true;
        }
        boolean isMetaDataRequest = request.isMetadataRequest();
        if (isMetaDataRequest && request.isReadOnly()) {
            return true;
        }
        return !isMetaDataRequest && !request.isAddressRefresh() && request.isReadOnly();
    }

    private ShouldRetryResult shouldRetryOnSessionNotAvailable(RxDocumentServiceRequest request) {
        ++this.sessionTokenRetryCount;
        if (!this.enableEndpointDiscovery) {
            return ShouldRetryResult.noRetry();
        }
        if (this.canUseMultipleWriteLocations) {
            UnmodifiableList<URI> endpoints;
            UnmodifiableList<URI> unmodifiableList = endpoints = this.isReadRequest ? this.globalEndpointManager.getApplicableReadEndpoints(request) : this.globalEndpointManager.getApplicableWriteEndpoints(request);
            if (this.sessionTokenRetryCount >= endpoints.size()) {
                return ShouldRetryResult.noRetry();
            }
            this.retryContext = new RetryContext(this.sessionTokenRetryCount, true);
            return ShouldRetryResult.retryAfter(Duration.ZERO);
        }
        if (this.sessionTokenRetryCount > 1) {
            return ShouldRetryResult.noRetry();
        }
        this.retryContext = new RetryContext(0, false);
        return ShouldRetryResult.retryAfter(Duration.ZERO);
    }

    private Mono<ShouldRetryResult> shouldRetryOnStaleContainer() {
        ++this.staleContainerRetryCount;
        if (this.rxCollectionCache == null || this.staleContainerRetryCount > 1) {
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        this.request.setForceNameCacheRefresh(true);
        if (this.request.intendedCollectionRidPassedIntoSDK) {
            return this.rxCollectionCache.refreshAsync(null, this.request).then(Mono.just((Object)ShouldRetryResult.noRetry()));
        }
        if (StringUtils.isNotEmpty(this.request.getHeaders().get("x-ms-cosmos-intended-collection-rid"))) {
            this.request.getHeaders().remove("x-ms-cosmos-intended-collection-rid");
        }
        return this.rxCollectionCache.refreshAsync(null, this.request).then(Mono.just((Object)ShouldRetryResult.retryAfter(Duration.ZERO)));
    }

    private Mono<ShouldRetryResult> shouldRetryOnEndpointFailureAsync(boolean isReadRequest, boolean forceRefresh, boolean usePreferredLocations) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        Mono<Void> refreshLocationCompletable = this.refreshLocation(isReadRequest, forceRefresh, usePreferredLocations);
        Duration retryDelay = Duration.ZERO;
        if (!isReadRequest) {
            logger.debug("Failover happening. retryCount {}", (Object)this.failoverRetryCount);
            if (this.failoverRetryCount > 1) {
                retryDelay = Duration.ofMillis(1000L);
            }
        } else {
            retryDelay = Duration.ofMillis(1000L);
        }
        return refreshLocationCompletable.then(Mono.just((Object)ShouldRetryResult.retryAfter(retryDelay)));
    }

    private Mono<ShouldRetryResult> shouldRetryOnGatewayTimeout() {
        boolean canFailoverOnTimeout = this.canGatewayRequestFailoverOnTimeout(this.request);
        if (canFailoverOnTimeout) {
            if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
                logger.warn("shouldRetryOnHttpTimeout() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
                return Mono.just((Object)ShouldRetryResult.noRetry());
            }
            ++this.failoverRetryCount;
            this.retryContext = new RetryContext(this.failoverRetryCount, true);
            Duration retryDelay = Duration.ofMillis(1000L);
            return Mono.just((Object)ShouldRetryResult.retryAfter(retryDelay));
        }
        return Mono.just((Object)ShouldRetryResult.NO_RETRY);
    }

    private Mono<ShouldRetryResult> shouldNotRetryOnEndpointFailureAsync(boolean isReadRequest, boolean forceRefresh, boolean usePreferredLocations) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        Mono<Void> refreshLocationCompletable = this.refreshLocation(isReadRequest, forceRefresh, usePreferredLocations);
        return refreshLocationCompletable.then(Mono.just((Object)ShouldRetryResult.noRetry()));
    }

    private Mono<Void> refreshLocation(boolean isReadRequest, boolean forceRefresh, boolean usePreferredLocations) {
        ++this.failoverRetryCount;
        if (isReadRequest) {
            logger.warn("marking the endpoint {} as unavailable for read", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForRead(this.locationEndpoint);
        } else {
            logger.warn("marking the endpoint {} as unavailable for write", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForWrite(this.locationEndpoint);
        }
        this.retryContext = new RetryContext(this.failoverRetryCount, usePreferredLocations);
        return this.globalEndpointManager.refreshLocationAsync(null, forceRefresh);
    }

    private Mono<ShouldRetryResult> shouldRetryOnBackendServiceUnavailableAsync(boolean isReadRequest, boolean isWebExceptionRetriable, boolean nonIdempotentWriteRetriesEnabled, CosmosException cosmosException) {
        if (!isReadRequest && !this.shouldRetryWriteOnServiceUnavailable(nonIdempotentWriteRetriesEnabled, isWebExceptionRetriable, cosmosException)) {
            logger.warn("shouldRetryOnBackendServiceUnavailableAsync() Not retrying on write with non retriable exception and non server returned service unavailable. Retry count = {}", (Object)this.serviceUnavailableRetryCount);
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        if (this.serviceUnavailableRetryCount++ > 1) {
            logger.warn("shouldRetryOnBackendServiceUnavailableAsync() Not retrying. Retry count = {}", (Object)this.serviceUnavailableRetryCount);
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        if (!this.canUseMultipleWriteLocations && !isReadRequest) {
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        int availablePreferredLocations = this.globalEndpointManager.getPreferredLocationCount();
        if (availablePreferredLocations <= 1) {
            logger.warn("shouldRetryOnServiceUnavailable() Not retrying. No other regions available for the request. AvailablePreferredLocations = {}", (Object)availablePreferredLocations);
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        logger.warn("shouldRetryOnServiceUnavailable() Retrying. Received on endpoint {}, IsReadRequest = {}", (Object)this.locationEndpoint, (Object)isReadRequest);
        this.retryContext = new RetryContext(this.serviceUnavailableRetryCount, true);
        return Mono.just((Object)ShouldRetryResult.retryAfter(Duration.ZERO));
    }

    @Override
    public void onBeforeSendRequest(RxDocumentServiceRequest request) {
        this.request = request;
        this.isReadRequest = request.isReadOnlyRequest();
        this.canUseMultipleWriteLocations = this.globalEndpointManager.canUseMultipleWriteLocations(request);
        if (request.requestContext != null) {
            request.requestContext.cosmosDiagnostics = this.cosmosDiagnostics;
        }
        if (request.requestContext != null) {
            request.requestContext.clearRouteToLocation();
        }
        if (this.retryContext != null) {
            request.requestContext.routeToLocation(this.retryContext.retryCount, this.retryContext.retryRequestOnPreferredLocations);
        }
        this.request.faultInjectionRequestContext = this.faultInjectionRequestContext;
        this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(request);
        if (request.requestContext != null) {
            request.requestContext.routeToLocation(this.locationEndpoint);
        }
    }

    @Override
    public com.azure.cosmos.implementation.RetryContext getRetryContext() {
        return BridgeInternal.getRetryContext(this.getCosmosDiagnostics());
    }

    public boolean canUsePreferredLocations() {
        return this.retryContext != null && this.retryContext.retryRequestOnPreferredLocations;
    }

    CosmosDiagnostics getCosmosDiagnostics() {
        return this.cosmosDiagnostics;
    }

    private boolean shouldRetryWriteOnServiceUnavailable(boolean nonIdempotentWriteRetriesEnabled, boolean isWebExceptionRetriable, CosmosException cosmosException) {
        if (nonIdempotentWriteRetriesEnabled || isWebExceptionRetriable) {
            return true;
        }
        if (cosmosException instanceof ServiceUnavailableException) {
            ServiceUnavailableException serviceUnavailableException = (ServiceUnavailableException)cosmosException;
            return serviceUnavailableException.getSubStatusCode() == 21008 || serviceUnavailableException.getSubStatusCode() == 21005;
        }
        return false;
    }

    private static class RetryContext {
        public int retryCount;
        public boolean retryRequestOnPreferredLocations;

        public RetryContext(int retryCount, boolean retryRequestOnPreferredLocations) {
            this.retryCount = retryCount;
            this.retryRequestOnPreferredLocations = retryRequestOnPreferredLocations;
        }
    }
}

