/*
 * Decompiled with CFR 0.152.
 */
package com.amazon.ws.emr.hadoop.fs.s3;

import com.amazon.ws.emr.hadoop.fs.files.TemporaryFiles;
import com.amazon.ws.emr.hadoop.fs.s3.S3ObjectRequestFactory;
import com.amazon.ws.emr.hadoop.fs.s3.lite.AmazonS3Lite;
import com.amazon.ws.emr.hadoop.fs.s3.upload.dispatch.AbortMultipartUploadEvent;
import com.amazon.ws.emr.hadoop.fs.s3.upload.dispatch.CompleteMultipartUploadEvent;
import com.amazon.ws.emr.hadoop.fs.s3.upload.plan.UploadPlan;
import com.amazon.ws.emr.hadoop.fs.s3n.ProgressableResettableBufferedFileInputStream;
import com.amazon.ws.emr.hadoop.fs.s3n.SpeculativeUtils;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.AmazonClientException;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.InitiateMultipartUploadResult;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.PartETag;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.UploadPartResult;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.annotations.VisibleForTesting;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.base.Preconditions;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.base.Strings;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.collect.Lists;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.FutureCallback;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.Futures;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.ListenableFuture;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.ListeningExecutorService;
import com.amazon.ws.emr.hadoop.fs.shaded.org.joda.time.DateTime;
import com.amazon.ws.emr.hadoop.fs.util.ConfigurationUtils;
import com.amazon.ws.emr.hadoop.fs.util.EmrFsUtils;
import com.amazon.ws.emr.hadoop.fs.util.Sleeper;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultipartUploadManager {
    private static final Logger logger = LoggerFactory.getLogger(MultipartUploadManager.class);
    private static final double TH_FRACTION_MAX_PART_SIZE = 0.7;
    private int nextPartNumber = 1;
    private double thFractionPartsCompleted;
    private double fractionPartAvgCompletionTime;
    private int partAttempts;
    private Set<Integer> incompletePartNums = Collections.synchronizedSet(new HashSet());
    private ConcurrentHashMap<Integer, List<MultiPartUploadFuture>> partNumFutureMap = new ConcurrentHashMap();
    private Object partNumFutureMapHandle = new Object();
    private UploadPlan uploadPlan;
    private String uploadId;
    private String serverSideEncryptionAlgorithm;
    private String serverSideEncryptionKmsKeyId;
    private AmazonS3Lite s3;
    private ListeningExecutorService executorService;
    private long totalLength = 0L;
    private long maxPartSize;
    private UploadPartResult thePartUploadedSynchronously = null;
    private Configuration conf;
    private Progressable progressable;
    private TemporaryFiles temporaryFiles;
    private Sleeper sleeper = Sleeper.defaultSleeper();
    private State state = State.INITIALIZED;

    @VisibleForTesting
    MultipartUploadManager() {
    }

    private void initializeFromConf(Configuration conf) {
        this.thFractionPartsCompleted = ConfigurationUtils.getPartsCompletedThresholdPercent(conf);
        this.fractionPartAvgCompletionTime = ConfigurationUtils.getFractionPartAvgCompletionTime(conf);
        this.partAttempts = ConfigurationUtils.getPartAttempts(conf);
    }

    public double getThFractionPartsCompleted() {
        return this.thFractionPartsCompleted;
    }

    public double getFractionPartAvgCompletionTime() {
        return this.fractionPartAvgCompletionTime;
    }

    public int getPartAttempts() {
        return this.partAttempts;
    }

    public void start() throws IOException {
        this.checkNotAlreadyStarted();
        this.uploadId = this.initiateMultipartUpload();
        this.state = State.IN_PROGRESS;
    }

    private String initiateMultipartUpload() throws IOException {
        ObjectMetadata md = new ObjectMetadata();
        if (!Strings.isNullOrEmpty(this.serverSideEncryptionAlgorithm)) {
            md.setSSEAlgorithm(this.serverSideEncryptionAlgorithm);
        }
        md.setContentType("binary/octet-stream");
        S3ObjectRequestFactory requestFactory = new S3ObjectRequestFactory(this.conf, this.serverSideEncryptionKmsKeyId);
        InitiateMultipartUploadRequest request = requestFactory.newMultipartUploadRequest(this.uploadPlan.getBucket(), this.uploadPlan.getKey());
        request.setObjectMetadata(md);
        return this.initiateUploadOrWrapExceptionIfFails(request).getUploadId();
    }

    private InitiateMultipartUploadResult initiateUploadOrWrapExceptionIfFails(InitiateMultipartUploadRequest request) throws IOException {
        try {
            return this.s3.initiateMultipartUpload(request);
        }
        catch (AmazonClientException e) {
            throw new IOException(e);
        }
    }

    public void abort() throws IOException {
        this.checkInProgress();
        this.state = State.ABORTED;
        this.cancelAllFutures();
        this.abortMultipartUpload();
    }

    private void cancelAllFutures() {
        for (Map.Entry<Integer, List<MultiPartUploadFuture>> entry : this.partNumFutureMap.entrySet()) {
            List<MultiPartUploadFuture> multiPartUploadFutures = entry.getValue();
            for (MultiPartUploadFuture multipartUploadFuture : multiPartUploadFutures) {
                Future<UploadPartResult> future = multipartUploadFuture.getFuture();
                future.cancel(true);
            }
        }
    }

    private void abortMultipartUpload() throws IOException {
        try {
            this.dispatchAbortEvent();
        }
        catch (AmazonClientException e) {
            throw new IOException(e);
        }
    }

    private void dispatchAbortEvent() throws IOException {
        this.uploadPlan.getMultipartDispatcher().abort(AbortMultipartUploadEvent.builder().bucket(this.uploadPlan.getBucket()).key(this.uploadPlan.getKey()).uploadId(this.uploadId).build());
    }

    private MultiPartUploadFuture createMultiPartUploadFuture(long partSize, MultipartUploadCallable multipartUploadCallable, int partNum) {
        ListenableFuture<UploadPartResult> future = this.executorService.submit(multipartUploadCallable);
        MultipartUploadFutureCallBack futureCallBack = new MultipartUploadFutureCallBack(future, partNum);
        Futures.addCallback(future, futureCallBack, this.executorService);
        MultiPartUploadFuture multiPartUploadFuture = new MultiPartUploadFuture(partSize, future, multipartUploadCallable);
        return multiPartUploadFuture;
    }

    @VisibleForTesting
    boolean shouldSpawnNewFuture(MultiPartUploadFuture remFuture) {
        if (remFuture.getStartTime() == null) {
            return false;
        }
        double avgCompletionTime = this.getAverageCompletionTime(this.partNumFutureMap, this.maxPartSize);
        if (avgCompletionTime == 0.0) {
            return false;
        }
        long timeSinceFutureStarted = DateTime.now().getMillis() - remFuture.getStartTime().getMillis();
        double thresholdTimeForSpawn = this.calculateThresholdTimeForSpawn(this.incompletePartNums.size(), this.partNumFutureMap.size(), avgCompletionTime);
        logger.debug("Time since incomplete future started: {}", (Object)timeSinceFutureStarted);
        logger.debug("Threshold time before spawn: {}", (Object)thresholdTimeForSpawn);
        if ((double)timeSinceFutureStarted > thresholdTimeForSpawn) {
            logger.debug("Incomplete future exceeded threshold, will start new one...");
            return true;
        }
        return false;
    }

    @VisibleForTesting
    double getAverageCompletionTime(ConcurrentHashMap<Integer, List<MultiPartUploadFuture>> partNumFutureMap, long partSize) {
        long totalTime = 0L;
        int completedFutures = 0;
        for (Map.Entry<Integer, List<MultiPartUploadFuture>> entry : partNumFutureMap.entrySet()) {
            MultiPartUploadFuture multipartUploadFuture;
            int partNum = entry.getKey();
            if (this.incompletePartNums.contains(partNum) || (multipartUploadFuture = entry.getValue().get(0)).getEndTime() == null || !((double)multipartUploadFuture.getPartSize() / (double)partSize >= 0.7)) continue;
            totalTime += multipartUploadFuture.getEndTime().getMillis() - multipartUploadFuture.getStartTime().getMillis();
            ++completedFutures;
        }
        if (completedFutures == 0) {
            return 0.0;
        }
        return (double)totalTime / (double)completedFutures;
    }

    @VisibleForTesting
    double calculateThresholdTimeForSpawn(int incompletePartSize, int totalPartSize, double avgCompletionTime) {
        double fractionPartsIncomplete = (double)incompletePartSize / (double)totalPartSize;
        return (1.0 + fractionPartsIncomplete) * (avgCompletionTime * this.fractionPartAvgCompletionTime);
    }

    private void spawnNewFutureIfNeeded(int partNum) throws IOException {
        List<MultiPartUploadFuture> multiPartUploadFutures = this.partNumFutureMap.get(partNum);
        logger.debug("Number of running attempts for: {} are: {}", (Object)partNum, (Object)multiPartUploadFutures.size());
        MultiPartUploadFuture multiPartUploadFuture = multiPartUploadFutures.get(multiPartUploadFutures.size() - 1);
        if (this.shouldSpawnNewFuture(multiPartUploadFuture)) {
            if (multiPartUploadFutures.size() >= this.partAttempts) {
                logger.error("Upload attempts for part num: {} have already reached max limit of: {}, will throw exception and fail", (Object)partNum, (Object)this.partAttempts);
                throw new IllegalStateException("Reached max limit of upload attempts for part");
            }
            logger.debug("Creating new future for partNum: {}", (Object)partNum);
            MultipartUploadCallable multipartUploadCallable = multiPartUploadFuture.getMultiPartUploadCallable();
            Path origPartFile = multipartUploadCallable.getPartFilePath();
            Path clonePartFile = SpeculativeUtils.clonePartFileIfNotAlreadyCompleted(this.temporaryFiles, origPartFile, partNum);
            if (clonePartFile == null) {
                return;
            }
            MultipartUploadCallable newMultiPartUploadCallable = new MultipartUploadCallable(multipartUploadCallable.getPartNumber(), clonePartFile);
            MultiPartUploadFuture newMultipartUploadFuture = this.createMultiPartUploadFuture(multiPartUploadFuture.getPartSize(), newMultiPartUploadCallable, partNum);
            multiPartUploadFutures.add(newMultipartUploadFuture);
        }
    }

    public void commit() throws IOException {
        this.checkInProgress();
        this.checkAtLeastOnePartAdded();
        try {
            this.waitUntilNoMoreIncompleteParts();
            List<PartETag> etags = this.collectPartEtags();
            this.dispatchCompleteEvent(etags);
            this.state = State.COMMITTED;
        }
        catch (Exception e) {
            logger.error("Error during multipart upload for bucket '{}' and key '{}'", new Object[]{this.uploadPlan.getBucket(), this.uploadPlan.getKey(), e});
            throw new IOException("Error during multipart upload", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitUntilNoMoreIncompleteParts() throws IOException, InterruptedException {
        int totalParts = this.partNumFutureMap.size();
        while (!this.incompletePartNums.isEmpty()) {
            int completedFutures = totalParts - this.incompletePartNums.size();
            if (completedFutures > 0 && (double)completedFutures >= this.thFractionPartsCompleted * (double)totalParts) {
                logger.debug("{} part(s) completed, checking heuristic...", (Object)completedFutures);
                Set<Integer> set = this.incompletePartNums;
                synchronized (set) {
                    for (int partNum : this.incompletePartNums) {
                        this.spawnNewFutureIfNeeded(partNum);
                    }
                }
            }
            this.sleeper.sleep(1000L);
        }
    }

    private void dispatchCompleteEvent(List<PartETag> etags) throws IOException {
        this.uploadPlan.getMultipartDispatcher().complete(CompleteMultipartUploadEvent.builder().bucket(this.uploadPlan.getBucket()).key(this.uploadPlan.getKey()).uploadId(this.uploadId).totalLength(this.totalLength).partETags(etags).extraUploadMetadata(this.uploadPlan.getExtraUploadMetadata()).build());
    }

    @VisibleForTesting
    List<PartETag> collectPartEtags() throws ExecutionException, InterruptedException {
        if (this.thePartUploadedSynchronously != null) {
            return Lists.newArrayList(this.thePartUploadedSynchronously.getPartETag());
        }
        return this.collectPartEtagsFromFutures();
    }

    private List<PartETag> collectPartEtagsFromFutures() throws ExecutionException, InterruptedException {
        ArrayList<PartETag> etags = new ArrayList<PartETag>();
        for (Map.Entry<Integer, List<MultiPartUploadFuture>> entry : this.partNumFutureMap.entrySet()) {
            MultiPartUploadFuture multiPartUploadFuture = entry.getValue().get(0);
            etags.add(multiPartUploadFuture.getFuture().get().getPartETag());
        }
        return etags;
    }

    public void addPartAsynchronously(Path contents) throws IOException {
        this.checkAddPartPreconditions();
        long length = Files.size(contents);
        this.startUploadingPartAsynchronously(this.nextPartNumber, contents, length);
        this.totalLength += length;
        ++this.nextPartNumber;
    }

    private void startUploadingPartAsynchronously(int partNumber, Path file, long length) {
        this.incompletePartNums.add(partNumber);
        MultipartUploadCallable callable = new MultipartUploadCallable(partNumber, file);
        MultiPartUploadFuture future = this.createMultiPartUploadFuture(length, callable, partNumber);
        if (this.incompletePartNums.contains(partNumber)) {
            this.addInitialFutureForPart(partNumber, future);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void addInitialFutureForPart(int partNumber, MultiPartUploadFuture future) {
        ArrayList<MultiPartUploadFuture> futures = Lists.newArrayList(future);
        Object object = this.partNumFutureMapHandle;
        synchronized (object) {
            this.partNumFutureMap.put(partNumber, futures);
            this.partNumFutureMapHandle.notifyAll();
        }
    }

    public void addOnlyPartSynchronously(Path contents) throws IOException {
        try {
            this.checkAddPartPreconditions();
            Preconditions.checkState(this.nextPartNumber == 1, "Cannot add part synchronously as we already added parts asynchronously");
            PartFile partFile = new PartFile(contents);
            Preconditions.checkArgument(partFile.length <= this.maxPartSize, "The part is too large to be handled synchronously");
            this.thePartUploadedSynchronously = this.uploadPart(this.nextPartNumber, partFile);
            this.totalLength += partFile.length;
            ++this.nextPartNumber;
        }
        catch (AmazonClientException e) {
            throw new IOException(e);
        }
        finally {
            this.temporaryFiles.delete(contents);
        }
    }

    public boolean isInProgress() {
        return this.state == State.IN_PROGRESS;
    }

    private boolean isCommitted() {
        return this.state == State.COMMITTED;
    }

    private void checkNotAlreadyStarted() {
        Preconditions.checkState(this.state == State.INITIALIZED, "Multipart upload manager has already been started");
    }

    private void checkAddPartPreconditions() {
        this.checkInProgress();
        Preconditions.checkState(this.thePartUploadedSynchronously == null, "The only part has already been added synchronously");
    }

    private void checkInProgress() {
        Preconditions.checkState(this.isInProgress(), "Multipart upload manager is not in progress");
    }

    private void checkAtLeastOnePartAdded() {
        Preconditions.checkState(this.nextPartNumber > 1, "At least one part must be added in order to commit");
    }

    @VisibleForTesting
    UploadPartResult getThePartUploadedSynchronously() {
        return this.thePartUploadedSynchronously;
    }

    @VisibleForTesting
    long getTotalLength() {
        return this.totalLength;
    }

    private UploadPartResult uploadPart(int partNumberToUse, PartFile partFile) throws IOException {
        try (InputStream input = this.openPartFileInputStream(partFile.path);){
            logger.debug("uploadPart using local file: {}", (Object)partFile);
            UploadPartResult uploadPartResult = this.s3.uploadPart(this.newUploadPartRequest(partNumberToUse, input, partFile.length));
            return uploadPartResult;
        }
    }

    private InputStream openPartFileInputStream(Path path) throws IOException {
        return new ProgressableResettableBufferedFileInputStream(path.toFile(), this.progressable);
    }

    private UploadPartRequest newUploadPartRequest(int partNumberToUse, InputStream partContent, long partFileSize) {
        return new UploadPartRequest().withBucketName(this.uploadPlan.getBucket()).withKey(this.uploadPlan.getKey()).withUploadId(this.uploadId).withPartNumber(partNumberToUse).withInputStream(partContent).withPartSize(partFileSize);
    }

    private class MultipartUploadCallable
    implements Callable<UploadPartResult> {
        private final int partNumber;
        private final Path partFilePath;
        private boolean shouldCallAbortOnCompletion;

        public MultipartUploadCallable(int partNumber, Path partFilePath) {
            this.partNumber = partNumber;
            this.partFilePath = partFilePath;
        }

        public void setShouldCallAbortOnCompletion(boolean shouldCallAbortOnCompletion) {
            this.shouldCallAbortOnCompletion = shouldCallAbortOnCompletion;
        }

        public int getPartNumber() {
            return this.partNumber;
        }

        public Path getPartFilePath() {
            return this.partFilePath;
        }

        @Override
        public UploadPartResult call() throws Exception {
            try {
                PartFile partFile = new PartFile(this.partFilePath);
                UploadPartResult result = this.doUpload(partFile);
                if (MultipartUploadManager.this.isCommitted() && this.shouldCallAbortOnCompletion) {
                    MultipartUploadManager.this.dispatchAbortEvent();
                }
                UploadPartResult uploadPartResult = result;
                return uploadPartResult;
            }
            catch (InterruptedException ie) {
                throw ie;
            }
            catch (Exception e) {
                logger.info("UploadPart error", (Throwable)e);
                throw e;
            }
            finally {
                MultipartUploadManager.this.temporaryFiles.delete(this.partFilePath);
            }
        }

        private UploadPartResult doUpload(PartFile partFile) throws InterruptedException, IOException {
            this.setActualStartTimeForLastAddedPartUploadFuture();
            return MultipartUploadManager.this.uploadPart(this.partNumber, partFile);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void setActualStartTimeForLastAddedPartUploadFuture() throws InterruptedException {
            Object object = MultipartUploadManager.this.partNumFutureMapHandle;
            synchronized (object) {
                try {
                    while (MultipartUploadManager.this.partNumFutureMap.get(this.partNumber) == null) {
                        logger.debug("Cannot get scheduled tasks for nextPartNumber {}, put thread to wait 100 ms or till it's notified by the addPartAsynchronously", (Object)this.partNumber);
                        MultipartUploadManager.this.partNumFutureMapHandle.wait(100L);
                    }
                }
                catch (InterruptedException ie) {
                    logger.debug("Part upload ({}, {}) was interrupted by another identical task.", (Object)this.partNumber, (Object)MultipartUploadManager.this.uploadId);
                    throw ie;
                }
            }
            List multiPartUploadFutures = (List)MultipartUploadManager.this.partNumFutureMap.get(this.partNumber);
            Optional<MultiPartUploadFuture> partUploadFuture = multiPartUploadFutures.stream().filter(partUploadAttempt -> partUploadAttempt.getMultiPartUploadCallable().getPartFilePath().equals(this.partFilePath)).findAny();
            if (partUploadFuture.isPresent()) {
                logger.debug("Setting upload start time for part {} for path {}", (Object)this.partNumber, (Object)this.partFilePath);
                partUploadFuture.get().setStartTime(DateTime.now());
            }
        }
    }

    private class MultipartUploadFutureCallBack
    implements FutureCallback<UploadPartResult> {
        private ListenableFuture<UploadPartResult> future;
        private int partNum;

        public MultipartUploadFutureCallBack(ListenableFuture<UploadPartResult> future, int partNum) {
            this.future = future;
            this.partNum = partNum;
        }

        @Override
        public void onFailure(Throwable arg0) {
            if (this.future.isCancelled()) {
                logger.debug("Multipart Upload for part: {} cancelled", (Object)this.partNum);
            } else {
                MultipartUploadManager.this.incompletePartNums.remove(this.partNum);
                logger.debug("Removed {} from incomplete partNum set because it failed", (Object)this.partNum);
                MultiPartUploadFuture multiPartUploadFuture = new MultiPartUploadFuture(0L, this.future, null);
                ArrayList<MultiPartUploadFuture> newMultipartUploadFutures = new ArrayList<MultiPartUploadFuture>();
                newMultipartUploadFutures.add(multiPartUploadFuture);
                MultipartUploadManager.this.partNumFutureMap.put(this.partNum, newMultipartUploadFutures);
                logger.debug("Added just the failed future to the list of futures for partNum: {}", (Object)this.partNum);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onSuccess(UploadPartResult arg0) {
            DateTime endTime = DateTime.now();
            Object object = MultipartUploadManager.this.partNumFutureMapHandle;
            synchronized (object) {
                try {
                    while (MultipartUploadManager.this.partNumFutureMap.get(this.partNum) == null) {
                        logger.debug("Waiting update from addPartAsynchronously for part uploading, but this should be rarely happened");
                        MultipartUploadManager.this.partNumFutureMapHandle.wait();
                    }
                }
                catch (InterruptedException e) {
                    throw new RuntimeException("Thread interrupted in multipart upload future callback's onSuccess", e);
                }
            }
            List multiPartUploadFutures = (List)MultipartUploadManager.this.partNumFutureMap.get(this.partNum);
            logger.debug("Total spawned multipart upload futures for partNum: {} are: {}", (Object)this.partNum, (Object)multiPartUploadFutures.size());
            ArrayList<MultiPartUploadFuture> newMultipartUploadFutures = null;
            for (MultiPartUploadFuture multiPartUploadFuture : multiPartUploadFutures) {
                if (multiPartUploadFuture.getFuture().isDone()) {
                    multiPartUploadFuture.setEndTime(endTime);
                    newMultipartUploadFutures = new ArrayList<MultiPartUploadFuture>();
                    newMultipartUploadFutures.add(multiPartUploadFuture);
                    continue;
                }
                long startTime = multiPartUploadFuture.getStartTime() == null ? endTime.getMillis() : multiPartUploadFuture.getStartTime().getMillis();
                logger.debug("Cancelling future for partNum: {} running for: {} s", (Object)this.partNum, (Object)((endTime.getMillis() - startTime) / 1000L));
                multiPartUploadFuture.getFuture().cancel(true);
                multiPartUploadFuture.getMultiPartUploadCallable().setShouldCallAbortOnCompletion(true);
            }
            if (newMultipartUploadFutures != null) {
                MultipartUploadManager.this.partNumFutureMap.put(this.partNum, newMultipartUploadFutures);
            }
            logger.debug("Going to remove {} from the incomplete part num set", (Object)this.partNum);
            MultipartUploadManager.this.incompletePartNums.remove(this.partNum);
        }
    }

    @VisibleForTesting
    class MultiPartUploadFuture {
        private long partSize;
        private DateTime startTime;
        private DateTime endTime;
        private Future<UploadPartResult> future;
        private MultipartUploadCallable multipartUploadCallable;

        public MultiPartUploadFuture(long partSize, Future<UploadPartResult> future, MultipartUploadCallable multiPartUploadCallable) {
            this.partSize = partSize;
            this.future = future;
            this.multipartUploadCallable = multiPartUploadCallable;
        }

        public long getPartSize() {
            return this.partSize;
        }

        public DateTime getStartTime() {
            return this.startTime;
        }

        public void setStartTime(DateTime startTime) {
            this.startTime = startTime;
        }

        public DateTime getEndTime() {
            return this.endTime;
        }

        public void setEndTime(DateTime endTime) {
            this.endTime = endTime;
        }

        public Future<UploadPartResult> getFuture() {
            return this.future;
        }

        public MultipartUploadCallable getMultiPartUploadCallable() {
            return this.multipartUploadCallable;
        }
    }

    private static final class PartFile {
        final Path path;
        final long length;

        PartFile(Path path) throws IOException {
            this(path, Files.size(path));
        }

        public String toString() {
            return "MultipartUploadManager.PartFile(path=" + this.path + ", length=" + this.length + ")";
        }

        public PartFile(Path path, long length) {
            this.path = path;
            this.length = length;
        }
    }

    public static class Builder {
        private MultipartUploadManager multipartUpload = new MultipartUploadManager();

        public Builder withServerSideEncryptionAlgorithm(String s) {
            this.multipartUpload.serverSideEncryptionAlgorithm = s;
            return this;
        }

        public Builder withServerSideKmsKeyId(String serverSideEncryptionKmsKeyId) {
            this.multipartUpload.serverSideEncryptionKmsKeyId = serverSideEncryptionKmsKeyId;
            return this;
        }

        public Builder withUploadPlan(UploadPlan uploadPlan) {
            this.multipartUpload.uploadPlan = uploadPlan;
            return this;
        }

        public Builder withUploadId(String s) {
            this.multipartUpload.uploadId = s;
            return this;
        }

        public Builder withProgressable(Progressable progressable) {
            this.multipartUpload.progressable = progressable;
            return this;
        }

        @VisibleForTesting
        Builder withS3(AmazonS3Lite s3, int maxRetries) {
            this.multipartUpload.s3 = EmrFsUtils.getAmazonS3LiteWithRetryPolicy(s3, maxRetries);
            return this;
        }

        public Builder withS3(AmazonS3Lite s3) {
            this.multipartUpload.s3 = EmrFsUtils.getAmazonS3LiteWithRetryPolicy(s3);
            return this;
        }

        public Builder withExecutorService(ListeningExecutorService executorService) {
            this.multipartUpload.executorService = executorService;
            return this;
        }

        public Builder withConf(Configuration conf) {
            this.multipartUpload.conf = conf;
            this.multipartUpload.initializeFromConf(conf);
            return this;
        }

        public Builder withMaxPartSize(long maxPartSize) {
            this.multipartUpload.maxPartSize = maxPartSize;
            return this;
        }

        public Builder withTemporaryFiles(TemporaryFiles temporaryFiles) {
            this.multipartUpload.temporaryFiles = temporaryFiles;
            return this;
        }

        @VisibleForTesting
        Builder withSleeper(Sleeper sleeper) {
            this.multipartUpload.sleeper = sleeper;
            return this;
        }

        public MultipartUploadManager build() {
            MultipartUploadManager newMultipartUpload = new MultipartUploadManager();
            newMultipartUpload.serverSideEncryptionAlgorithm = this.multipartUpload.serverSideEncryptionAlgorithm;
            newMultipartUpload.serverSideEncryptionKmsKeyId = this.multipartUpload.serverSideEncryptionKmsKeyId;
            newMultipartUpload.uploadId = this.multipartUpload.uploadId;
            newMultipartUpload.progressable = this.multipartUpload.progressable;
            newMultipartUpload.uploadPlan = this.multipartUpload.uploadPlan;
            newMultipartUpload.s3 = this.multipartUpload.s3;
            newMultipartUpload.executorService = this.multipartUpload.executorService;
            newMultipartUpload.thFractionPartsCompleted = this.multipartUpload.thFractionPartsCompleted;
            newMultipartUpload.fractionPartAvgCompletionTime = this.multipartUpload.fractionPartAvgCompletionTime;
            newMultipartUpload.partAttempts = this.multipartUpload.partAttempts;
            newMultipartUpload.maxPartSize = this.multipartUpload.maxPartSize;
            newMultipartUpload.conf = this.multipartUpload.conf;
            newMultipartUpload.temporaryFiles = this.multipartUpload.temporaryFiles;
            newMultipartUpload.sleeper = this.multipartUpload.sleeper;
            return newMultipartUpload;
        }
    }

    private static enum State {
        INITIALIZED,
        IN_PROGRESS,
        ABORTED,
        COMMITTED;

    }
}

