/*
 * Decompiled with CFR 0.152.
 */
package com.amazon.ws.emr.hadoop.fs.maintenance;

import com.amazon.ws.emr.hadoop.fs.s3.lite.AmazonS3Lite;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.ListMultipartUploadsRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.MultipartUpload;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.MultipartUploadListing;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.annotations.VisibleForTesting;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.MoreExecutors;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.amazon.ws.emr.hadoop.fs.util.S3UriUtils;
import java.net.URI;
import java.time.Duration;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultipartUploadCleaner {
    private static final Logger logger = LoggerFactory.getLogger(MultipartUploadCleaner.class);
    private static final long SCHEDULER_SHUTDOWN_TIMEOUT_MILLIS = 1000L;
    private static final int MULTIPART_LISTING_PAGE_SIZE = 1000;
    private static MultipartUploadCleaner INSTANCE = null;
    final Duration cleanerThreadFixedDelay = Duration.ofSeconds(900L);
    final int maxJitterDelayMillis;
    private final Duration incompleteMultipartAgeThreshold;
    private final Duration cleanerThreadInitialDelay = Duration.ofSeconds(0L);
    private final Set<String> s3Buckets = new HashSet<String>();
    private final AmazonS3Lite s3;
    private final Optional<ScheduledExecutorService> optionalScheduler;
    private final Random random;
    private volatile boolean isClosed = false;

    @VisibleForTesting
    MultipartUploadCleaner(Configuration conf, AmazonS3Lite s3) {
        this.incompleteMultipartAgeThreshold = Duration.ofSeconds(conf.getLong("fs.s3.multipart.clean.age.threshold", 604800L));
        this.maxJitterDelayMillis = conf.getInt("fs.s3.multipart.clean.jitter.max", 10000);
        this.s3 = s3;
        boolean isMultipartCleanupEnabled = conf.getBoolean("fs.s3.multipart.clean.enabled", false);
        this.optionalScheduler = isMultipartCleanupEnabled ? Optional.of(Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("multipart-cleanup-service").setDaemon(true).build())) : Optional.empty();
        this.random = new Random();
    }

    public static synchronized MultipartUploadCleaner getInstance(Configuration conf, AmazonS3Lite s3) {
        if (INSTANCE == null) {
            INSTANCE = new MultipartUploadCleaner(conf, s3);
            Runtime.getRuntime().addShutdownHook(new Thread(() -> INSTANCE.close()));
        }
        return INSTANCE;
    }

    private static Future<?> scheduleMultipartCleanupAsync(ScheduledExecutorService scheduler, Runnable scheduleCleanupTask, Consumer<Exception> errorHandler) {
        return scheduler.submit(() -> {
            try {
                scheduleCleanupTask.run();
            }
            catch (Exception e) {
                errorHandler.accept(e);
            }
        });
    }

    private boolean isMultipartCleanupEnabled() {
        return this.optionalScheduler.isPresent();
    }

    private Duration getJitterDelay() {
        return Duration.ofMillis(this.random.nextInt(this.maxJitterDelayMillis));
    }

    @VisibleForTesting
    synchronized void doScheduleMultipartCleanup(String bucketName) {
        if (this.isMultipartCleanupEnabled() && !this.s3Buckets.contains(bucketName)) {
            this.s3Buckets.add(bucketName);
            this.scheduleCleanUpWithJitter(this.cleanerThreadInitialDelay, new MultipartCleanUpRunnable(bucketName));
            logger.info("Multipart upload cleanup is now enabled for bucket: {}", (Object)bucketName);
        }
    }

    public Optional<Future<?>> scheduleMultipartCleanup(String bucketName) {
        return this.optionalScheduler.map(scheduler -> MultipartUploadCleaner.scheduleMultipartCleanupAsync(scheduler, () -> this.doScheduleMultipartCleanup(bucketName), e -> this.logWarnIfNotClosed("Exception scheduling multipart upload cleanup for bucket " + bucketName, (Throwable)e)));
    }

    public Optional<Future<?>> scheduleMultipartCleanup(Path path) {
        return this.optionalScheduler.map(scheduler -> MultipartUploadCleaner.scheduleMultipartCleanupAsync(scheduler, () -> this.doScheduleMultipartCleanup(S3UriUtils.pathToBucket(path)), e -> this.logWarnIfNotClosed("Exception scheduling multipart upload cleanup for path: " + path, (Throwable)e)));
    }

    public Optional<Future<?>> scheduleMultipartCleanup(URI uri) {
        return this.optionalScheduler.map(scheduler -> MultipartUploadCleaner.scheduleMultipartCleanupAsync(scheduler, () -> this.doScheduleMultipartCleanup(S3UriUtils.pathToBucket(new Path(uri))), e -> this.logWarnIfNotClosed("Exception scheduling multipart upload cleanup for uri: " + uri, (Throwable)e)));
    }

    private void logWarnIfNotClosed(String message, Throwable throwable) {
        if (!this.isClosed) {
            logger.warn(message, throwable);
        } else {
            logger.debug("Error while shutting down the multipart upload cleanup system: {}", (Object)message, (Object)throwable);
        }
    }

    @VisibleForTesting
    Set<String> getS3Buckets() {
        return this.s3Buckets;
    }

    public synchronized void close() {
        if (!this.isClosed) {
            this.isClosed = true;
            this.optionalScheduler.ifPresent(scheduler -> {
                try {
                    logger.info("Shutting down multipart cleanup service");
                    MoreExecutors.shutdownAndAwaitTermination(scheduler, 1000L, TimeUnit.MILLISECONDS);
                }
                catch (Exception ex) {
                    logger.error("Couldn't shutdown multipart cleanup service", (Throwable)ex);
                }
            });
        }
    }

    @VisibleForTesting
    int abortMultipartUploads(List<MultipartUpload> multipartUploads, Date cleanBefore, String bucketName) {
        int numUploadsAborted = 0;
        for (MultipartUpload upload : multipartUploads) {
            if (!upload.getInitiated().before(cleanBefore)) continue;
            logger.info("Aborting MultipartUpload {} for {} as it is older than configured age threshold of {} seconds", new Object[]{upload.getUploadId(), upload.getKey(), this.incompleteMultipartAgeThreshold.getSeconds()});
            try {
                this.s3.abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, upload.getKey(), upload.getUploadId()));
                ++numUploadsAborted;
            }
            catch (Exception ex) {
                this.logWarnIfNotClosed("Exception caught while aborting multipart uploads", ex);
            }
        }
        return numUploadsAborted;
    }

    private void scheduleCleanUpWithJitter(Duration delay, Runnable cleaner) {
        try {
            this.optionalScheduler.ifPresent(scheduler -> {
                if (!this.isClosed) {
                    scheduler.schedule(cleaner, delay.plus(this.getJitterDelay()).toMillis(), TimeUnit.MILLISECONDS);
                }
            });
        }
        catch (Exception ex) {
            this.logWarnIfNotClosed("Exception caught while trying to schedule cleanup task", ex);
        }
    }

    @VisibleForTesting
    class MultipartCleanUpRunnable
    implements Runnable {
        private final String bucketName;
        private final String prefix;
        private final Duration nextRoundDelay;
        private final int multipartListingPageSize;
        private Date cleanBefore;
        private MultipartUploadListing uploadListing;

        MultipartCleanUpRunnable(String bucketName) {
            this(bucketName, null, this$0.cleanerThreadFixedDelay, 1000);
        }

        private void scheduleNextCleanupRound() {
            logger.debug("Scheduling next cleanup round for bucket {} in approximately {} seconds", (Object)this.bucketName, (Object)this.nextRoundDelay.getSeconds());
            this.cleanBefore = null;
            this.uploadListing = null;
            MultipartUploadCleaner.this.scheduleCleanUpWithJitter(this.nextRoundDelay, this);
        }

        private void scheduleNextPageCleanup() {
            logger.debug("Scheduling cleanup for next page of multipart uploads for bucket {} in approximately 0 seconds", (Object)this.bucketName);
            MultipartUploadCleaner.this.scheduleCleanUpWithJitter(Duration.ZERO, this);
        }

        @Override
        public void run() {
            try {
                if (this.cleanBefore == null) {
                    this.cleanBefore = new Date(new Date().getTime() - MultipartUploadCleaner.this.incompleteMultipartAgeThreshold.toMillis());
                }
                if (this.uploadListing == null) {
                    logger.debug("Cleaning multipart uploads before {} for bucket {}", (Object)this.cleanBefore, (Object)this.bucketName);
                    this.uploadListing = MultipartUploadCleaner.this.s3.listMultipartUploads(this.newListMultipartUploadsRequest());
                } else {
                    logger.debug("Cleaning next page of multipart uploads before {} for bucket {}", (Object)this.cleanBefore, (Object)this.bucketName);
                    ListMultipartUploadsRequest request = this.newListMultipartUploadsRequest().withKeyMarker(this.uploadListing.getNextKeyMarker()).withUploadIdMarker(this.uploadListing.getNextUploadIdMarker());
                    this.uploadListing = MultipartUploadCleaner.this.s3.listMultipartUploads(request);
                }
                MultipartUploadCleaner.this.abortMultipartUploads(this.uploadListing.getMultipartUploads(), this.cleanBefore, this.bucketName);
                if (this.uploadListing.isTruncated()) {
                    this.scheduleNextPageCleanup();
                } else {
                    this.scheduleNextCleanupRound();
                }
            }
            catch (Exception ex) {
                MultipartUploadCleaner.this.logWarnIfNotClosed("Exception caught while clearing multipart uploads", ex);
                this.scheduleNextCleanupRound();
            }
        }

        private ListMultipartUploadsRequest newListMultipartUploadsRequest() {
            return new ListMultipartUploadsRequest(this.bucketName).withPrefix(this.prefix).withMaxUploads(this.multipartListingPageSize);
        }

        public MultipartCleanUpRunnable(String bucketName, String prefix, Duration nextRoundDelay, int multipartListingPageSize) {
            this.bucketName = bucketName;
            this.prefix = prefix;
            this.nextRoundDelay = nextRoundDelay;
            this.multipartListingPageSize = multipartListingPageSize;
        }
    }
}

