/*
 * Decompiled with CFR 0.152.
 */
package com.amazon.ws.emr.hadoop.fs.s3;

import com.amazon.ws.emr.hadoop.fs.EmrFSFutureCallback;
import com.amazon.ws.emr.hadoop.fs.cse.CSEUtils;
import com.amazon.ws.emr.hadoop.fs.s3.DefaultDestinationObjectMetadataFactory;
import com.amazon.ws.emr.hadoop.fs.s3.DestinationObjectMetadataFactory;
import com.amazon.ws.emr.hadoop.fs.s3.S3ObjectRequestFactory;
import com.amazon.ws.emr.hadoop.fs.s3.lite.AmazonS3Lite;
import com.amazon.ws.emr.hadoop.fs.s3.lite.ConsistencyExceptionThrowableObjectMetadataRetriever;
import com.amazon.ws.emr.hadoop.fs.s3.upload.dispatch.UploadObserver;
import com.amazon.ws.emr.hadoop.fs.s3.upload.dispatch.UploadObserverContext;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.AmazonClientException;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.AmazonServiceException;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CompleteMultipartUploadResult;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CopyObjectResult;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CopyPartRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.CopyPartResult;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.InitiateMultipartUploadResult;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.PartETag;
import com.amazon.ws.emr.hadoop.fs.shaded.com.amazonaws.services.s3.model.S3ObjectId;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.base.Optional;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.base.Preconditions;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.base.Strings;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.collect.Lists;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.ListenableFuture;
import com.amazon.ws.emr.hadoop.fs.shaded.com.google.common.util.concurrent.ListeningExecutorService;
import com.amazon.ws.emr.hadoop.fs.util.ConfigurationUtils;
import com.amazon.ws.emr.hadoop.fs.util.EmrFsUtils;
import java.io.IOException;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.retry.RetryInvocationHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultipartCopyManager {
    public static final Logger LOG = LoggerFactory.getLogger(MultipartCopyManager.class);
    private final long partSize;
    private String srcBucketName;
    private String srcKey;
    private String dstBucketName;
    private String dstKey;
    private CannedAccessControlList acl;
    private AmazonS3Lite s3;
    private ListeningExecutorService executorService;
    private S3ObjectRequestFactory s3ObjectRequestFactory;
    private Configuration conf;
    private UploadObserver uploadObserver;
    private DestinationObjectMetadataFactory objectMetadataFactory;
    private ConsistencyExceptionThrowableObjectMetadataRetriever objectMetadataRetriever;

    private MultipartCopyManager(Builder builder) {
        this.srcBucketName = builder.srcBucketName;
        this.srcKey = builder.srcKey;
        this.dstBucketName = builder.dstBucketName;
        this.dstKey = builder.dstKey;
        this.acl = builder.acl;
        this.s3 = builder.s3;
        this.executorService = builder.executorService;
        this.conf = builder.conf;
        this.s3ObjectRequestFactory = new S3ObjectRequestFactory(this.conf, builder.serverSideEncryptionKmsKeyId);
        this.partSize = ConfigurationUtils.getDefaultPartSize(this.conf);
        this.uploadObserver = builder.uploadObserver;
        this.objectMetadataRetriever = builder.objectMetadataRetriever;
        this.objectMetadataFactory = new DefaultDestinationObjectMetadataFactory(builder.serverSideEncryptionAlgorithm, builder.serverSideEncryptionKmsKeyId);
    }

    private ObjectMetadata newObjectMetadataFromSrc(ObjectMetadata source) {
        return this.objectMetadataFactory.createFromSource(source);
    }

    private ObjectMetadata getObjectMetadata(String bucket, String key) {
        try {
            return this.s3.getObjectMetadata(bucket, key);
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                return null;
            }
            throw e;
        }
    }

    private Optional<Callable<String>> createCopyInstFileCallable(String srcBucketName, String srcKey, String dstBucketName, String dstKey) {
        String srcInstKey = new S3ObjectId(srcBucketName, srcKey).instructionFileId().getKey();
        String dstInstKey = new S3ObjectId(dstBucketName, dstKey).instructionFileId().getKey();
        ObjectMetadata metadata = this.getObjectMetadata(srcBucketName, srcInstKey);
        if (metadata != null) {
            return Optional.of(this.createCopyFileCallable(srcBucketName, srcInstKey, dstBucketName, dstInstKey, metadata));
        }
        return Optional.absent();
    }

    private void deleteObjectInstFile(String bucket, String key) {
        try {
            CSEUtils.deletePreviousInstructionFileIfNecessary(this.conf, this.s3, bucket, key, true);
        }
        catch (AmazonClientException e) {
            String instKey = new S3ObjectId(bucket, key).instructionFileId().getKey();
            LOG.warn("Failed to delete instruction file {}/{}", (Object)bucket, (Object)instKey);
        }
    }

    private Callable<String> createCopyFileCallable(final String srcBucketName, final String srcKey, final String dstBucketName, final String dstKey, final ObjectMetadata metadata) {
        return new Callable<String>(){

            @Override
            public String call() throws Exception {
                try {
                    LOG.debug("Copying from '{}/{}' to '{}/{}' using s3 copy", new Object[]{srcBucketName, srcKey, dstBucketName, dstKey});
                    CopyObjectRequest copyObjectRequest = MultipartCopyManager.this.s3ObjectRequestFactory.newCopyObjectRequest(srcBucketName, srcKey, dstBucketName, dstKey);
                    if (MultipartCopyManager.this.acl != null) {
                        copyObjectRequest.setCannedAccessControlList(MultipartCopyManager.this.acl);
                    }
                    copyObjectRequest.setNewObjectMetadata(metadata);
                    MultipartCopyManager.this.copyObject(copyObjectRequest);
                    return dstKey;
                }
                catch (AmazonClientException e) {
                    if (ConfigurationUtils.isClientSideEncryptionEnabled(MultipartCopyManager.this.conf)) {
                        MultipartCopyManager.this.deleteObjectInstFile(dstBucketName, dstKey);
                    }
                    throw e;
                }
            }
        };
    }

    public void copy() throws IOException {
        for (Callable<String> callable : this.createCopyCallables()) {
            try {
                callable.call();
            }
            catch (Exception e) {
                throw new IOException(e);
            }
        }
    }

    public List<Callable<String>> createCopyCallables() throws IOException {
        ObjectMetadata srcObjectMetadata = this.objectMetadataRetriever != null ? this.objectMetadataRetriever.get(EmrFsUtils.getPathForS3Object(this.srcBucketName, this.srcKey)) : this.s3.getObjectMetadata(this.srcBucketName, this.srcKey);
        long objectSize = srcObjectMetadata.getContentLength();
        ArrayList<Callable<String>> callables = Lists.newArrayList();
        if (ConfigurationUtils.isClientSideEncryptionEnabled(this.conf)) {
            callables.addAll(this.createCopyInstFileCallable(this.srcBucketName, this.srcKey, this.dstBucketName, this.dstKey).asSet());
        }
        if (objectSize <= this.partSize) {
            callables.add(this.createCopyFileCallable(this.srcBucketName, this.srcKey, this.dstBucketName, this.dstKey, this.newObjectMetadataFromSrc(srcObjectMetadata)));
        } else {
            try {
                LOG.debug("Copying from '{}/{}' to '{}/{}' using s3 multipart upload", new Object[]{this.srcBucketName, this.srcKey, this.dstBucketName, this.dstKey});
                ObjectMetadata dstObjectMetadata = this.newObjectMetadataFromSrc(srcObjectMetadata);
                InitiateMultipartUploadRequest initiateRequest = this.s3ObjectRequestFactory.newMultipartUploadRequest(this.dstBucketName, this.dstKey);
                initiateRequest.setObjectMetadata(dstObjectMetadata);
                if (this.acl != null) {
                    initiateRequest.setCannedACL(this.acl);
                }
                InitiateMultipartUploadResult initiateResult = this.s3.initiateMultipartUpload(initiateRequest);
                final String uploadId = initiateResult.getUploadId();
                final ArrayList<ListenableFuture<CopyPartResult>> copyPartFutures = Lists.newArrayList();
                long bytePosition = 0L;
                EmrFSFutureCallback<CopyPartResult> emrFSFutureCallback = new EmrFSFutureCallback<CopyPartResult>(true);
                int i = 1;
                while (bytePosition < objectSize) {
                    final CopyPartRequest copyPartRequest = new CopyPartRequest().withDestinationBucketName(this.dstBucketName).withDestinationKey(this.dstKey).withSourceBucketName(this.srcBucketName).withSourceKey(this.srcKey).withUploadId(uploadId).withFirstByte(bytePosition).withLastByte(bytePosition + Math.min(this.partSize - 1L, objectSize - bytePosition - 1L)).withPartNumber(i);
                    ListenableFuture<CopyPartResult> future = this.executorService.submit(new Callable<CopyPartResult>(){

                        @Override
                        public CopyPartResult call() throws Exception {
                            return MultipartCopyManager.this.s3.copyPart(copyPartRequest);
                        }
                    });
                    copyPartFutures.add(future);
                    emrFSFutureCallback.registerFuture(future);
                    bytePosition += Math.min(this.partSize, objectSize - bytePosition);
                    ++i;
                }
                callables.add(new Callable<String>(){

                    @Override
                    public String call() throws Exception {
                        try {
                            ArrayList<PartETag> partETags = Lists.newArrayList();
                            int i = 0;
                            for (Future copyPartFuture : copyPartFutures) {
                                boolean log = false;
                                if (!copyPartFuture.isDone()) {
                                    log = true;
                                    LOG.debug("Waiting for part " + i);
                                }
                                CopyPartResult copyPartResult = (CopyPartResult)copyPartFuture.get();
                                if (log) {
                                    LOG.debug("Done waiting for part " + copyPartResult.getPartNumber());
                                }
                                partETags.add(copyPartResult.getPartETag());
                            }
                            MultipartCopyManager.this.completeMultipartUpload(new CompleteMultipartUploadRequest(MultipartCopyManager.this.dstBucketName, MultipartCopyManager.this.dstKey, uploadId, partETags));
                        }
                        catch (InterruptedException | ExecutionException e) {
                            LOG.error(String.format("Failed to copy from '%s/%s' to '%s/%s'. Trying to abort multipart upload: %s", MultipartCopyManager.this.srcBucketName, MultipartCopyManager.this.srcKey, MultipartCopyManager.this.dstBucketName, MultipartCopyManager.this.dstKey, uploadId), (Throwable)e);
                            AbortMultipartUploadRequest request = new AbortMultipartUploadRequest(MultipartCopyManager.this.dstBucketName, MultipartCopyManager.this.dstKey, uploadId);
                            try {
                                MultipartCopyManager.this.s3.abortMultipartUpload(request);
                                LOG.info(String.format("Aborted multipart upload: %s", uploadId));
                            }
                            catch (AmazonClientException amazonClientException) {
                                LOG.error(String.format("Failed to abort multipart upload: %s", uploadId), (Throwable)amazonClientException);
                            }
                            throw new IOException(e);
                        }
                        return MultipartCopyManager.this.dstKey;
                    }
                });
            }
            catch (AmazonClientException e) {
                if (ConfigurationUtils.isClientSideEncryptionEnabled(this.conf)) {
                    this.deleteObjectInstFile(this.dstBucketName, this.dstKey);
                }
                throw e;
            }
        }
        return callables;
    }

    private void copyObject(CopyObjectRequest request) throws IOException {
        UploadObserverContext observerContext = new UploadObserverContext(request.getDestinationBucketName(), request.getDestinationKey());
        this.uploadObserver.beforeUploadCompletion(observerContext);
        CopyObjectResult result = this.s3.copyObject(request);
        observerContext.getStore().put(UploadObserverContext.ETAG_VERIFICATION_KEY, result.getETag());
        this.uploadObserver.afterUploadCompletion(observerContext);
    }

    private void completeMultipartUpload(CompleteMultipartUploadRequest request) throws IOException {
        UploadObserverContext observerContext = new UploadObserverContext(request.getBucketName(), request.getKey());
        this.uploadObserver.beforeUploadCompletion(observerContext);
        CompleteMultipartUploadResult result = this.s3.completeMultipartUpload(request);
        observerContext.getStore().put(UploadObserverContext.ETAG_VERIFICATION_KEY, result.getETag());
        this.uploadObserver.afterUploadCompletion(observerContext);
    }

    public static class Builder {
        private String srcBucketName;
        private String srcKey;
        private String dstBucketName;
        private String dstKey;
        private CannedAccessControlList acl;
        private AmazonS3Lite s3;
        private ListeningExecutorService executorService;
        private String serverSideEncryptionAlgorithm;
        private String serverSideEncryptionKmsKeyId;
        private Configuration conf;
        private UploadObserver uploadObserver = UploadObserver.none();
        private ConsistencyExceptionThrowableObjectMetadataRetriever objectMetadataRetriever;

        public Builder withS3(AmazonS3Lite s3) {
            this.s3 = s3;
            return this;
        }

        public Builder withExecutorService(ListeningExecutorService ExecutorService2) {
            this.executorService = ExecutorService2;
            return this;
        }

        public Builder withSrcBucketName(String srcBucketName) {
            this.srcBucketName = srcBucketName;
            return this;
        }

        public Builder withSrcKey(String srcKey) {
            this.srcKey = srcKey;
            return this;
        }

        public Builder withDstBucketName(String dstBucketName) {
            this.dstBucketName = dstBucketName;
            return this;
        }

        public Builder withDstKey(String dstKey) {
            this.dstKey = dstKey;
            return this;
        }

        public Builder withAcl(CannedAccessControlList acl) {
            this.acl = acl;
            return this;
        }

        public Builder withServerSideEncryption(String serverSideEncryptionAlgorithm) {
            this.serverSideEncryptionAlgorithm = serverSideEncryptionAlgorithm;
            return this;
        }

        public Builder withServerSideKmsKeyId(String serverSideEncryptionKmsKeyId) {
            this.serverSideEncryptionKmsKeyId = serverSideEncryptionKmsKeyId;
            return this;
        }

        public Builder withConf(Configuration conf) {
            this.conf = conf;
            return this;
        }

        public Builder withUploadObserver(UploadObserver uploadObserver) {
            this.uploadObserver = uploadObserver;
            return this;
        }

        public Builder withObjectMetadataRetriever(ConsistencyExceptionThrowableObjectMetadataRetriever objectMetadataRetriever) {
            this.objectMetadataRetriever = objectMetadataRetriever;
            return this;
        }

        public MultipartCopyManager build() {
            if (this.objectMetadataRetriever != null) {
                Preconditions.checkArgument(Proxy.isProxyClass(this.objectMetadataRetriever.getClass()));
                Preconditions.checkArgument(Proxy.getInvocationHandler(this.objectMetadataRetriever) instanceof RetryInvocationHandler);
            }
            Preconditions.checkNotNull(this.s3, "Amazons3 cannot be null");
            Preconditions.checkNotNull(this.executorService, "ExecutorService cannot be null");
            Preconditions.checkNotNull(this.uploadObserver, "UploadObserver cannot be null");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(this.srcBucketName), "Source bucket cannot be empty");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(this.srcKey), "Source key cannot be empty");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(this.dstBucketName), "Destination bucket cannot be empty");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(this.dstKey), "Destination key cannot be empty");
            return new MultipartCopyManager(this);
        }
    }
}

