/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.dynamodb;

import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Callable;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.dynamodb.PrintCounter;
import org.apache.hadoop.mapred.Reporter;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.ReadableDuration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkException;

public class DynamoDBFibonacciRetryer {
    private static final Log log = LogFactory.getLog(DynamoDBFibonacciRetryer.class);
    private static final Set<Integer> internalErrorStatusCodes = new HashSet<Integer>();
    private static final Set<String> throttleErrorCodes = new HashSet<String>();
    private final Duration retryPeriod;
    private final Random random = new Random(System.currentTimeMillis());
    private volatile boolean isShutdown;
    private int fib1 = 0;
    private int fib2 = 1;
    private int retryCount;

    public DynamoDBFibonacciRetryer(Duration retryPeriod) {
        this.retryPeriod = retryPeriod;
    }

    public <T> RetryResult<T> runWithRetry(Callable<T> callable, Reporter reporter, PrintCounter retryCounter) {
        this.fib1 = 0;
        this.fib2 = 1;
        this.retryCount = 0;
        DateTime currentTime = new DateTime(DateTimeZone.UTC);
        DateTime retryEndTime = currentTime.plus((ReadableDuration)this.retryPeriod);
        while (true) {
            if (this.isShutdown) {
                log.info((Object)"Is shut down, giving up and returning null");
                return null;
            }
            try {
                T returnObj = callable.call();
                return new RetryResult<T>(returnObj, this.retryCount);
            }
            catch (Exception e) {
                this.handleException(retryEndTime, e, reporter, retryCounter);
                continue;
            }
            break;
        }
    }

    public synchronized void shutdown() {
        if (this.isShutdown) {
            throw new IllegalStateException("Cannot call shutdown() more than once");
        }
        this.isShutdown = true;
    }

    private void handleException(DateTime retryEndTime, Exception exception, Reporter reporter, PrintCounter retryCounter) {
        DateTime currentTime = new DateTime(DateTimeZone.UTC);
        long maxDelay = retryEndTime.getMillis() - currentTime.getMillis();
        if (this.verifyRetriableException(exception) && maxDelay > 0L) {
            AwsServiceException ase;
            if (exception instanceof AwsServiceException && !throttleErrorCodes.contains((ase = (AwsServiceException)exception).awsErrorDetails().errorCode()) && !internalErrorStatusCodes.contains(ase.awsErrorDetails().sdkHttpResponse().statusCode())) {
                throw new RuntimeException(exception);
            }
            this.incrementRetryCounter(reporter, retryCounter);
            ++this.retryCount;
            log.warn((Object)("Retry: " + this.retryCount + " Exception: " + exception));
            this.delayOp(maxDelay);
        } else if (this.isShutdown) {
            log.warn((Object)"Retries exceeded and caught, but is shutdown so not throwing", (Throwable)exception);
        } else {
            log.error((Object)("Retries exceeded or non-retryable exception, throwing: " + exception));
            throw new RuntimeException(exception);
        }
    }

    private boolean verifyRetriableException(Exception exception) {
        return exception instanceof AwsServiceException || exception instanceof SdkException || exception instanceof SocketException || exception instanceof SocketTimeoutException;
    }

    private void incrementRetryCounter(Reporter reporter, PrintCounter retryCounter) {
        if (reporter != null) {
            if (retryCounter != null) {
                reporter.incrCounter(retryCounter.getGroup(), retryCounter.getName(), 1L);
            } else {
                reporter.progress();
            }
        }
    }

    private void delayOp(long maxDelay) {
        this.increment();
        try {
            long computedDelay = this.fib2 * 50 + this.random.nextInt(this.fib1 * 100);
            long delay = Math.min(computedDelay, maxDelay + (long)this.random.nextInt(100));
            Thread.sleep(delay);
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while retrying", e);
        }
    }

    private void increment() {
        int sum = this.fib1 + this.fib2;
        this.fib1 = this.fib2;
        this.fib2 = sum;
    }

    static {
        internalErrorStatusCodes.add(500);
        internalErrorStatusCodes.add(503);
        throttleErrorCodes.add("ProvisionedThroughputExceededException");
        throttleErrorCodes.add("ThrottlingException");
    }

    public static class RetryResult<T> {
        public final T result;
        public final int retries;

        public RetryResult(T result, int retries) {
            this.result = result;
            this.retries = retries;
        }
    }
}

