/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.dynamodb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.primitives.Ints;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.dynamodb.CredentialPairName;
import org.apache.hadoop.dynamodb.DynamoDBFibonacciRetryer;
import org.apache.hadoop.dynamodb.DynamoDBUtil;
import org.apache.hadoop.dynamodb.PrintCounter;
import org.apache.hadoop.dynamodb.filter.DynamoDBIndexInfo;
import org.apache.hadoop.dynamodb.filter.DynamoDBQueryFilter;
import org.apache.hadoop.mapred.Reporter;
import org.joda.time.Duration;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProviderChain;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.http.apache.ApacheHttpClient;
import software.amazon.awssdk.http.apache.ProxyConfiguration;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.dynamodb.DynamoDbClient;
import software.amazon.awssdk.services.dynamodb.DynamoDbClientBuilder;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemResponse;
import software.amazon.awssdk.services.dynamodb.model.Capacity;
import software.amazon.awssdk.services.dynamodb.model.Condition;
import software.amazon.awssdk.services.dynamodb.model.ConsumedCapacity;
import software.amazon.awssdk.services.dynamodb.model.DeleteRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.PutRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryResponse;
import software.amazon.awssdk.services.dynamodb.model.ReturnConsumedCapacity;
import software.amazon.awssdk.services.dynamodb.model.ScanRequest;
import software.amazon.awssdk.services.dynamodb.model.ScanResponse;
import software.amazon.awssdk.services.dynamodb.model.TableDescription;
import software.amazon.awssdk.services.dynamodb.model.WriteRequest;

public class DynamoDBClient {
    private static final Log log = LogFactory.getLog(DynamoDBClient.class);
    private static final int DEFAULT_RETRY_DURATION = 10;
    private static final long MAX_BACKOFF_IN_MILLISECONDS = 3000L;
    private static final CredentialPairName DYNAMODB_CREDENTIAL_PAIR_NAME = new CredentialPairName("dynamodb.awsAccessKeyId", "dynamodb.awsSecretAccessKey");
    private static final CredentialPairName DYNAMODB_SESSION_CREDENTIAL_PAIR_NAME = new CredentialPairName(DYNAMODB_CREDENTIAL_PAIR_NAME.getAccessKeyName(), DYNAMODB_CREDENTIAL_PAIR_NAME.getSecretKeyName(), "dynamodb.awsSessionToken");
    private static final CredentialPairName DEFAULT_CREDENTIAL_PAIR_NAME = new CredentialPairName("fs.s3.awsAccessKeyId", "fs.s3.awsSecretAccessKey");
    private final Map<String, List<WriteRequest>> writeBatchMap = new HashMap<String, List<WriteRequest>>();
    private final DynamoDbClient dynamoDB;
    private int writeBatchMapSizeBytes;
    private int batchWriteRetries;
    private final Configuration config;
    private final long maxBatchSize;
    private final long maxItemByteSize;

    public DynamoDBClient() {
        this((DynamoDbClient)null, null);
    }

    public DynamoDBClient(DynamoDbClient amazonDynamoDBClient, Configuration conf) {
        this.dynamoDB = amazonDynamoDBClient;
        this.config = conf;
        this.maxBatchSize = 0x1000000L;
        this.maxItemByteSize = 409600L;
    }

    public DynamoDBClient(Configuration conf) {
        this(conf, null);
    }

    public DynamoDBClient(Configuration conf, String region) {
        Preconditions.checkNotNull((Object)conf, (Object)"conf cannot be null.");
        this.config = conf;
        this.dynamoDB = this.getDynamoDBClient(conf, region);
        this.maxBatchSize = this.config.getLong("dynamodb.max.batch.size", 0x1000000L);
        this.maxItemByteSize = this.config.getLong("dynamodb.max.item.size", 409600L);
    }

    public final Map<String, List<WriteRequest>> getWriteBatchMap() {
        return this.writeBatchMap;
    }

    public TableDescription describeTable(String tableName) {
        DescribeTableRequest describeTablesRequest = (DescribeTableRequest)DescribeTableRequest.builder().tableName(tableName).build();
        try {
            DynamoDBFibonacciRetryer.RetryResult<DescribeTableResponse> describeResult = this.getRetryDriver().runWithRetry(() -> {
                DescribeTableResponse response = this.dynamoDB.describeTable(describeTablesRequest);
                log.info((Object)("Describe table output: " + response));
                return response;
            }, null, null);
            return ((DescribeTableResponse)describeResult.result).table();
        }
        catch (Exception e) {
            throw new RuntimeException("Could not lookup table " + tableName + " in DynamoDB.", e);
        }
    }

    public DynamoDBFibonacciRetryer.RetryResult<ScanResponse> scanTable(String tableName, DynamoDBQueryFilter dynamoDBQueryFilter, Integer segment, Integer totalSegments, Map<String, AttributeValue> exclusiveStartKey, long limit, Reporter reporter) {
        Map<String, Condition> scanFilter;
        ScanRequest.Builder scanRequestBuilder = ScanRequest.builder().tableName(tableName).exclusiveStartKey(exclusiveStartKey).limit(Integer.valueOf(Ints.checkedCast((long)limit))).segment(segment).totalSegments(totalSegments).returnConsumedCapacity(ReturnConsumedCapacity.TOTAL);
        if (dynamoDBQueryFilter != null && !(scanFilter = dynamoDBQueryFilter.getScanFilter()).isEmpty()) {
            scanRequestBuilder.scanFilter(scanFilter);
        }
        ScanRequest scanRequest = (ScanRequest)scanRequestBuilder.build();
        DynamoDBFibonacciRetryer.RetryResult<ScanResponse> retryResult = this.getRetryDriver().runWithRetry(() -> {
            log.debug((Object)("Executing DynamoDB scan: " + scanRequest));
            return this.dynamoDB.scan(scanRequest);
        }, reporter, PrintCounter.DynamoDBReadThrottle);
        return retryResult;
    }

    public DynamoDBFibonacciRetryer.RetryResult<QueryResponse> queryTable(String tableName, DynamoDBQueryFilter dynamoDBQueryFilter, Map<String, AttributeValue> exclusiveStartKey, long limit, Reporter reporter) {
        QueryRequest.Builder queryRequestBuilder = QueryRequest.builder().tableName(tableName).exclusiveStartKey(exclusiveStartKey).keyConditions(dynamoDBQueryFilter.getKeyConditions()).limit(Integer.valueOf(Ints.checkedCast((long)limit))).returnConsumedCapacity(ReturnConsumedCapacity.TOTAL);
        DynamoDBIndexInfo index = dynamoDBQueryFilter.getIndex();
        if (index != null) {
            log.debug((Object)("Using DynamoDB index: " + index.getIndexName()));
            queryRequestBuilder.indexName(index.getIndexName());
        }
        QueryRequest queryRequest = (QueryRequest)queryRequestBuilder.build();
        DynamoDBFibonacciRetryer.RetryResult<QueryResponse> retryResult = this.getRetryDriver().runWithRetry(() -> {
            log.debug((Object)("Executing DynamoDB query: " + queryRequest));
            return this.dynamoDB.query(queryRequest);
        }, reporter, PrintCounter.DynamoDBReadThrottle);
        return retryResult;
    }

    public BatchWriteItemResponse putBatch(String tableName, Map<String, AttributeValue> item, long maxItemsPerBatch, Reporter reporter, boolean deletionMode) throws UnsupportedEncodingException {
        List<Object> writeBatchList;
        int itemSizeBytes = DynamoDBUtil.getItemSizeBytes(item);
        if ((long)itemSizeBytes > this.maxItemByteSize) {
            throw new RuntimeException("Cannot pass items with size greater than " + this.maxItemByteSize + ". Item with size of " + itemSizeBytes + " was given.");
        }
        maxItemsPerBatch = DynamoDBUtil.getBoundedBatchLimit(this.config, maxItemsPerBatch);
        BatchWriteItemResponse response = null;
        if (this.writeBatchMap.containsKey(tableName)) {
            boolean totalSizeOfWriteBatchesOverLimit;
            boolean writeRequestsForTableAtLimit = (long)this.writeBatchMap.get(tableName).size() >= maxItemsPerBatch;
            boolean bl = totalSizeOfWriteBatchesOverLimit = (long)(this.writeBatchMapSizeBytes + itemSizeBytes) > this.maxBatchSize;
            if (writeRequestsForTableAtLimit || totalSizeOfWriteBatchesOverLimit) {
                response = this.writeBatch(reporter, itemSizeBytes);
            }
        }
        if (!this.writeBatchMap.containsKey(tableName)) {
            writeBatchList = new ArrayList((int)maxItemsPerBatch);
            this.writeBatchMap.put(tableName, writeBatchList);
        } else {
            writeBatchList = this.writeBatchMap.get(tableName);
        }
        log.info((Object)("BatchWriteItem deletionMode " + deletionMode));
        if (deletionMode) {
            writeBatchList.add(WriteRequest.builder().deleteRequest((DeleteRequest)DeleteRequest.builder().key(this.getKeys(item)).build()).build());
        } else {
            writeBatchList.add(WriteRequest.builder().putRequest((PutRequest)PutRequest.builder().item(item).build()).build());
        }
        this.writeBatchMapSizeBytes += itemSizeBytes;
        return response;
    }

    public void close() {
        while (!this.writeBatchMap.isEmpty()) {
            this.writeBatch(Reporter.NULL, 0);
        }
        if (this.dynamoDB != null) {
            this.dynamoDB.close();
        }
    }

    private Map<String, AttributeValue> getKeys(Map<String, AttributeValue> item) {
        String tableKeyNames = this.config.get("dynamodb.table.keyNames");
        if (tableKeyNames == null || tableKeyNames.isEmpty()) {
            return item;
        }
        HashSet<String> keySet = new HashSet<String>(Arrays.asList(tableKeyNames.split(",")));
        Map<String, AttributeValue> keys = item.entrySet().stream().filter(entry -> keySet.contains(entry.getKey())).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        if (keys.isEmpty()) {
            throw new IllegalArgumentException(String.format("Given item does not contain any key for the table: %s", tableKeyNames));
        }
        return keys;
    }

    private static Map<String, AttributeValue> getItemFromRequest(WriteRequest request) {
        if (request.putRequest() != null) {
            return request.putRequest().item();
        }
        return request.deleteRequest().key();
    }

    private BatchWriteItemResponse writeBatch(Reporter reporter, final int roomNeeded) {
        final BatchWriteItemRequest batchWriteItemRequest = (BatchWriteItemRequest)BatchWriteItemRequest.builder().requestItems(this.writeBatchMap).returnConsumedCapacity(ReturnConsumedCapacity.INDEXES).build();
        DynamoDBFibonacciRetryer.RetryResult<BatchWriteItemResponse> retryResult = this.getRetryDriver().runWithRetry(new Callable<BatchWriteItemResponse>(){

            @Override
            public BatchWriteItemResponse call() throws UnsupportedEncodingException, InterruptedException {
                DynamoDBClient.this.pauseExponentially(DynamoDBClient.this.batchWriteRetries);
                BatchWriteItemResponse result = DynamoDBClient.this.dynamoDB.batchWriteItem(batchWriteItemRequest);
                Map unprocessedItems = result.unprocessedItems();
                if (unprocessedItems == null || unprocessedItems.isEmpty()) {
                    DynamoDBClient.this.batchWriteRetries = 0;
                } else {
                    DynamoDBClient.this.batchWriteRetries++;
                    int unprocessedItemCount = 0;
                    for (List unprocessedWriteRequests : unprocessedItems.values()) {
                        unprocessedItemCount += unprocessedWriteRequests.size();
                        int batchSizeBytes = 0;
                        for (Object request : unprocessedWriteRequests) {
                            batchSizeBytes += DynamoDBUtil.getItemSizeBytes(DynamoDBClient.getItemFromRequest((WriteRequest)request));
                        }
                        long maxItemsPerBatch = DynamoDBClient.this.config.getLong("dynamodb.max.batch.items", 25L);
                        long maxBatchSize = DynamoDBClient.this.config.getLong("dynamodb.max.batch.size", 0x1000000L);
                        if ((long)unprocessedWriteRequests.size() < maxItemsPerBatch && maxBatchSize - (long)batchSizeBytes >= (long)roomNeeded) continue;
                        throw SdkException.builder().message("Full list of write requests not processed").build();
                    }
                    double consumed = 0.0;
                    for (ConsumedCapacity consumedCapacity : result.consumedCapacity()) {
                        consumed = consumedCapacity.table().capacityUnits();
                        if (consumedCapacity.localSecondaryIndexes() == null) continue;
                        for (Capacity lsiConsumedCapacity : consumedCapacity.localSecondaryIndexes().values()) {
                            consumed += lsiConsumedCapacity.capacityUnits().doubleValue();
                        }
                    }
                    int batchSize = 0;
                    for (List writeRequests : batchWriteItemRequest.requestItems().values()) {
                        batchSize += writeRequests.size();
                    }
                    log.debug((Object)("BatchWriteItem attempted " + batchSize + " items, consumed " + consumed + " wcu, left unprocessed " + unprocessedItemCount + " items, now at " + DynamoDBClient.this.batchWriteRetries + " retries"));
                }
                return result;
            }
        }, reporter, PrintCounter.DynamoDBWriteThrottle);
        this.writeBatchMap.clear();
        this.writeBatchMapSizeBytes = 0;
        Map unprocessedItems = ((BatchWriteItemResponse)retryResult.result).unprocessedItems();
        for (Map.Entry entry : unprocessedItems.entrySet()) {
            String key = (String)entry.getKey();
            List requests = (List)entry.getValue();
            for (WriteRequest request : requests) {
                this.writeBatchMapSizeBytes += DynamoDBUtil.getItemSizeBytes(DynamoDBClient.getItemFromRequest(request));
            }
            this.writeBatchMap.put(key, new ArrayList(requests));
        }
        return (BatchWriteItemResponse)retryResult.result;
    }

    private DynamoDBFibonacciRetryer getRetryDriver() {
        return new DynamoDBFibonacciRetryer(Duration.standardMinutes((long)10L));
    }

    private void pauseExponentially(int retries) throws InterruptedException {
        if (retries == 0) {
            return;
        }
        long scaleFactor = 500 + new Random().nextInt(100);
        long delay = (long)(Math.pow(2.0, retries) * (double)scaleFactor) / 4L;
        delay = Math.min(delay, 3000L);
        log.info((Object)("Pausing " + delay + " ms at retry " + retries));
        Thread.sleep(delay);
    }

    private DynamoDbClient getDynamoDBClient(Configuration conf, String region) {
        DynamoDbClientBuilder dynamoDbClientBuilder = DynamoDbClient.builder();
        dynamoDbClientBuilder.region(Region.of((String)DynamoDBUtil.getDynamoDBRegion(conf, region)));
        String customEndpoint = DynamoDBUtil.getDynamoDBEndpoint(conf, region);
        if (!Strings.isNullOrEmpty((String)customEndpoint)) {
            dynamoDbClientBuilder.endpointOverride(URI.create(customEndpoint));
        }
        return (DynamoDbClient)((DynamoDbClientBuilder)((DynamoDbClientBuilder)((DynamoDbClientBuilder)dynamoDbClientBuilder.httpClient(ApacheHttpClient.builder().proxyConfiguration(this.applyProxyConfiguration(conf)).build())).credentialsProvider(this.getAwsCredentialsProvider(conf))).overrideConfiguration((ClientOverrideConfiguration)ClientOverrideConfiguration.builder().retryPolicy(builder -> builder.numRetries(Integer.valueOf(1))).build())).build();
    }

    @VisibleForTesting
    ProxyConfiguration applyProxyConfiguration(Configuration conf) {
        ProxyConfiguration.Builder builder = ProxyConfiguration.builder();
        String proxyHost = conf.get("dynamodb.proxy.hostname");
        int proxyPort = conf.getInt("dynamodb.proxy.port", 0);
        String proxyUsername = conf.get("dynamodb.proxy.username");
        String proxyPassword = conf.get("dynamodb.proxy.password");
        boolean proxyHostAndPortPresent = false;
        if (!Strings.isNullOrEmpty((String)proxyHost) && proxyPort > 0) {
            builder.endpoint(this.buildProxyEndpoint(proxyHost, proxyPort));
            proxyHostAndPortPresent = true;
        } else if (Strings.isNullOrEmpty((String)proxyHost) ^ proxyPort <= 0) {
            throw new RuntimeException("Only one of proxy host and port are set, when both are required");
        }
        if (!Strings.isNullOrEmpty((String)proxyUsername) && !Strings.isNullOrEmpty((String)proxyPassword)) {
            if (!proxyHostAndPortPresent) {
                throw new RuntimeException("Proxy host and port must be supplied if proxy username and password are present");
            }
            builder.username(proxyUsername).password(proxyPassword);
        } else if (Strings.isNullOrEmpty((String)proxyUsername) ^ Strings.isNullOrEmpty((String)proxyPassword)) {
            throw new RuntimeException("Only one of proxy username and password are set, when both are required");
        }
        return (ProxyConfiguration)builder.build();
    }

    protected AwsCredentialsProvider getAwsCredentialsProvider(Configuration conf) {
        AwsBasicCredentials credentials;
        String sessionKey;
        String secretKey;
        String accessKey;
        ArrayList<Object> providersList = new ArrayList<Object>();
        String providerClass = conf.get("dynamodb.customAWSCredentialsProvider");
        if (!Strings.isNullOrEmpty((String)providerClass)) {
            providersList.add(DynamoDBUtil.loadAwsCredentialsProvider(providerClass, conf));
        }
        if (Strings.isNullOrEmpty((String)(accessKey = conf.get(DYNAMODB_SESSION_CREDENTIAL_PAIR_NAME.getAccessKeyName())))) {
            accessKey = conf.get(DEFAULT_CREDENTIAL_PAIR_NAME.getAccessKeyName());
            secretKey = conf.get(DEFAULT_CREDENTIAL_PAIR_NAME.getSecretKeyName());
            sessionKey = null;
        } else {
            secretKey = conf.get(DYNAMODB_SESSION_CREDENTIAL_PAIR_NAME.getSecretKeyName());
            sessionKey = conf.get(DYNAMODB_SESSION_CREDENTIAL_PAIR_NAME.getSessionKeyName());
        }
        if (Strings.isNullOrEmpty((String)accessKey) || Strings.isNullOrEmpty((String)secretKey)) {
            log.debug((Object)"Custom credential provider not found, loading default provider from sdk");
            providersList.add(DefaultCredentialsProvider.create());
        } else if (!Strings.isNullOrEmpty((String)sessionKey)) {
            credentials = AwsSessionCredentials.create((String)accessKey, (String)secretKey, (String)sessionKey);
            providersList.add(() -> DynamoDBClient.lambda$getAwsCredentialsProvider$5((AwsCredentials)credentials));
        } else {
            credentials = AwsBasicCredentials.create((String)accessKey, (String)secretKey);
            providersList.add(() -> DynamoDBClient.lambda$getAwsCredentialsProvider$6((AwsCredentials)credentials));
        }
        AwsCredentialsProvider[] providerArray = providersList.toArray(new AwsCredentialsProvider[providersList.size()]);
        AwsCredentialsProviderChain providerChain = AwsCredentialsProviderChain.builder().credentialsProviders(providerArray).reuseLastProviderEnabled(Boolean.valueOf(true)).build();
        return providerChain;
    }

    private URI buildProxyEndpoint(String proxyHost, int proxyPort) {
        String HTTP_PROTOCOL = "http://";
        return URI.create("http://" + proxyHost + ":" + proxyPort);
    }

    private static /* synthetic */ AwsCredentials lambda$getAwsCredentialsProvider$6(AwsCredentials credentials) {
        return credentials;
    }

    private static /* synthetic */ AwsCredentials lambda$getAwsCredentialsProvider$5(AwsCredentials credentials) {
        return credentials;
    }
}

